(ns ksql.transport.http
  (:require [cheshire.core :as json]
            [clojure.core.async :as async]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [ksql.compiler :as compiler]))

(def content-type
  "The default content type."
  "application/vnd.ksql.v1+json")

(def headers
  "The default HTTP headers."
  {"Accept" content-type
   "accept-encoding" "*"
   "Content-Type" content-type})

(def streaming-op?
  "Returns true if the given op keyword returns a streaming response."
  #{:print :query})

(def defaults
  "The clj-http defaults."
  {;; :accept :json
   ;; :debug true
   ;; :coerce :always
   ;; :content-type :json
   :headers headers
   :scheme :http
   :server-name "localhost"
   :server-port 8088
   :query-params {:timeout "5000"}})

(defn client [& opts]
  (merge defaults opts))

(defn- as
  "Returns the uri of the KSQL request."
  [{:keys [op] :as ast}]
  (if (streaming-op? op)
    :stream :auto))

(defn- body
  "Returns the body of the KSQL request."
  [ast]
  (json/generate-string
   {:ksql (str (first (compiler/sql ast)) ";")
    :streamsProperties
    {;; "ksql.streams.auto.offset.reset" "earliest"
     }}))

(defn- uri
  "Returns the uri of the KSQL request."
  [{:keys [op] :as ast}]
  (if (streaming-op? op)
    "/query" "/ksql"))

(defn request
  "Returns the Ring compatible KSQL request map."
  [statement]
  (let [ast (compiler/ast statement)]
    (-> {:as (as ast)
         :body (body ast)
         :request-method :post
         :uri (uri ast)}
        (merge (-> ast :db :transport)))))

(defn- success-response
  "Returns a successful response."
  [statement {:keys [body] :as response}]
  (if (instance? clojure.lang.IMeta body)
    (with-meta body (dissoc response :body))
    (-> body io/reader line-seq async/to-chan)))

(defn- parse-line [line]
  (try (json/parse-string line keyword)
       (catch Exception e
         (throw (ex-info (str "Can't parse line: " line) {:line line} e)))))

(defn log-line [x]
  (prn "LOG: " x)
  x)

(defn- massage-line [s]
  (-> (str/replace s #"^\[" "")
      (str/replace #"}]\s*$" "}")))

(def response-xf
  (comp (remove str/blank?)
        (map massage-line)
        (map parse-line)))
