(ns ksql.core
  (:refer-clojure :exclude [group-by list print])
  (:require [cats.context :as ctx]
            [cats.core :as m]
            [cats.monad.state :as state]
            [clojure.pprint :refer [pprint]]
            [clojure.spec.alpha :as s]
            [ksql.ast :as ast]
            [ksql.compiler :as compiler]
            [ksql.statement :refer [map->Statement statement?]]
            [ksql.transport.http :as http]
            [ksql.transport.aleph :as aleph]))

(def db
  "The default KSQL db."
  {:evaluator #'aleph/request!
   :transport http/defaults})

(s/def ::alias map?)
(s/def ::body (s/* ::clause))
(s/def ::clause state/state?)
(s/def ::column keyword?)
(s/def ::columns (s/coll-of ::column))
(s/def ::condition list?)
(s/def ::connector keyword?)
(s/def ::db map?)
(s/def ::expression list?)
(s/def ::join-type #{:left :inner :right})
(s/def ::result-materialization #{:changes})
(s/def ::source (s/or :alias? ::alias :keyword keyword?))
(s/def ::statement statement?)
(s/def ::stream keyword?)
(s/def ::table keyword?)

(defn- statement
  "Return a KSQL statement."
  [node & [body]]
  (map->Statement
   (m/mlet [_ (state/put node)
            _ (m/sequence (filter state/state? body))]
     (m/return node))))

(s/fdef statement
  :args (s/cat :node map? :body (s/* any?))
  :ret ::statement)

(defn- update-state [f & args]
  (m/mlet [state' (state/swap #(apply f % args))]
    (m/return state')))

(s/fdef update-state
  :args (s/cat :f ifn? :args (s/* any?))
  :ret state/state?)

(defn- assoc-node [node]
  (update-state assoc (get node :op) node))

(s/fdef assoc-node
  :args (s/cat :node map?)
  :ret state/state?)

(defn ast
  "Compile the `statement` to an AST."
  [statement]
  (last (seq (state/run (ctx/with-context state/context statement) {}))))

(s/fdef ast
  :args (s/cat :statement ::statement)
  :ret map?)

(defn pp
  "Compile the `statement` to an AST."
  [statement]
  (pprint (ast statement)))

(s/fdef pp
  :args (s/cat :statement ::statement))

(defn as
  ([alias]
   (assoc-node (ast/as (ast alias))))
  ([expr alias]
   (ast/as expr alias)))

(defn emit
  [result-materialization]
  (assoc-node (ast/emit result-materialization)))

(s/fdef emit
  :args (s/cat :result-materialization ::result-materialization)
  :ret ::clause)

(defn sql
  "Compile the `statement` to SQL string.."
  [statement]
  (compiler/sql statement))

(s/fdef sql
  :args (s/cat :statement ::statement)
  :ret string?)

(defn create-sink-connector
  {:style/indent 2}
  [db name & body]
  (statement (ast/create-sink-connector db name) body))

(s/fdef create-sink-connector
  :args (s/cat :db ::db :name ::connector :body ::body)
  :ret ::statement)

(defn create-source-connector
  {:style/indent 2}
  [db name & body]
  (statement (ast/create-source-connector db name) body))

(s/fdef create-source-connector
  :args (s/cat :db ::db :name ::connector :body ::body)
  :ret ::statement)

(defn create-stream
  {:style/indent 2}
  [db stream & body]
  (statement (ast/create-stream db stream body) body))

(s/fdef create-stream
  :args (s/alt :ary-2 (s/cat :db ::db :stream ::stream :body ::body)
               :ary-3 (s/cat :db ::db :stream ::stream :definition vector? :body ::body))
  :ret ::statement)

(defn create-table
  {:style/indent 2}
  [db stream & body]
  (statement (ast/create-table db stream body) body))

(s/fdef create-table
  :args (s/alt :ary-2 (s/cat :db ::db :table ::table :body ::body)
               :ary-3 (s/cat :db ::db :table ::table :definition vector? :body ::body))
  :ret ::statement)

(defn delete-topic
  ([]
   (delete-topic true))
  ([delete-topic?]
   (assoc-node (ast/delete-topic delete-topic?))))

(s/fdef delete-topic
  :args (s/alt :ary-0 (s/cat) :ary-1 (s/cat :delete-topic? (s/nilable boolean?)))
  :ret ::clause)

(defn drop-connector
  {:style/indent 2}
  [db connector]
  (statement (ast/drop-connector db connector)))

(s/fdef drop-connector
  :args (s/cat :db ::db :name ::connector)
  :ret ::statement)

(defn drop-table
  {:style/indent 2}
  [db table & body]
  (statement (ast/drop-table db table) body))

(s/fdef drop-table
  :args (s/cat :db ::db :table ::table :body ::body)
  :ret ::statement)

(defn drop-stream
  {:style/indent 2}
  [db stream & body]
  (statement (ast/drop-stream db stream) body))

(s/fdef drop-stream
  :args (s/cat :db ::db :stream ::stream :body ::body)
  :ret ::statement)

(defn from [source]
  (assoc-node (ast/from source)))

(s/fdef from
  :args (s/cat :source ::source)
  :ret ::clause)

(defn group-by [& columns]
  (assoc-node (ast/group-by columns)))

(s/fdef group-by
  :args (s/cat :columns (s/* ::column))
  :ret ::clause)

(defn join [source condition type]
  (assoc-node (ast/join source condition type)))

(s/fdef join
  :args (s/cat :source ::source :condition ::condition :type (s/nilable ::join-type))
  :ret ::clause)

(defn if-exists
  ([]
   (if-exists true))
  ([if-exists?]
   (assoc-node (ast/if-exists if-exists?))))

(s/fdef if-exists
  :args (s/alt :ary-0 (s/cat) :ary-1 (s/cat :if-exists? (s/nilable boolean?)))
  :ret ::clause)

(defn insert
  "Build a INSERT statement."
  {:style/indent 3}
  [db table columns & body]
  (statement (ast/insert db table columns) body))

(s/fdef insert
  :args (s/cat :db ::db :table ::table :columns ::columns :body ::body)
  :ret ::statement)

(defn order-by [& columns]
  (assoc-node (ast/order-by columns)))

(s/fdef order-by
  :args (s/cat :columns (s/* ::column))
  :ret ::clause)

(defn select
  {:style/indent 2}
  [db expr & body]
  (statement (ast/query db expr) body))

(s/fdef select
  :args (s/cat :db ::db :expr vector? :body ::body)
  :ret ::statement)

(defn show
  {:style/indent 2}
  [db type]
  (statement (ast/show db type)))

(s/fdef show
  :args (s/cat :db ::db :type keyword?)
  :ret ::statement)

(defn limit [limit]
  (assoc-node (ast/limit limit)))

(s/fdef limit
  :args (s/cat :limit nat-int?)
  :ret ::clause)

(defn list
  {:style/indent 2}
  [db type]
  (statement (ast/list db type)))

(s/fdef list
  :args (s/cat :db ::db :type keyword?)
  :ret ::statement)

(defn print
  {:style/indent 2}
  [db clause]
  (statement (ast/print db clause)))

(s/fdef print
  :args (s/cat :db ::db :clause any?)
  :ret ::statement)

(defn terminate
  {:style/indent 2}
  [db query]
  (statement (ast/terminate db query)))

(s/fdef terminate
  :args (s/cat :db ::db :query keyword?)
  :ret ::statement)

(defn values
  [values]
  (m/mlet [ast (state/get)
           :let [columns (map :identifier (-> ast :columns :elements))
                 node (ast/values columns values)]
           _ (state/swap #(assoc % :values node))]
    (m/return node)))

(defn where [expr]
  (assoc-node (ast/where expr)))

(s/fdef where
  :args (s/cat :expr ::expression)
  :ret ::clause)

(defn window [type & expressions]
  (assoc-node (ast/window type expressions)))

(s/fdef window
  :args (s/cat :type keyword? :expressions (s/* ::expression))
  :ret ::clause)

(defn with [opts]
  (assoc-node (ast/with opts)))

(s/fdef with
  :args (s/cat :opts map?)
  :ret ::clause)
