(ns ksql.ast
  (:refer-clojure :exclude [group-by list print])
  (:require [ksql.ast.children :as children]
            [ksql.ast.spec :as spec]))

(declare expression)

(defn identifier? [x]
  (keyword? x))

(defn identifier [expr]
  {:op :identifier
   :children children/identifier
   :identifier expr})

(defn literal? [x]
  (or (boolean? x)
      (nil? x)
      (number? x)
      (string? x)))

(defn function-call? [x]
  (and (list? x) (symbol? (first x))))

(defn function-call [expr]
  {:op :function-call
   :children children/function-call
   :name (first expr)
   :args (mapv expression (rest expr))})

(defn limit [limit]
  {:op :limit
   :children children/limit
   :limit limit})

(defn literal [expr]
  {:op :literal
   :children children/literal
   :literal expr})

(defn list [db type]
  {:op :list
   :children children/list
   :db db
   :type type})

(defn expression [expr]
  {:op :expression
   :children children/expression
   :expression
   (cond
     (identifier? expr)
     (identifier expr)
     (literal? expr)
     (literal expr)
     (function-call? expr)
     (function-call expr))})

(defn as
  ([alias]
   {:op :as
    :children children/as
    :alias alias})
  ([expr alias]
   {:op :as
    :children children/as
    :expression (expression expr)
    :alias (identifier alias)}))

(defn result-materialization [result-materialization]
  {:op :result-materialization
   :children children/result-materialization
   :result-materialization result-materialization})

(defn emit [materialization]
  {:op :emit
   :children children/emit
   :result-materialization (result-materialization materialization)})

(defn delete-topic [delete-topic]
  {:op :delete-topic
   :children children/delete-topic
   :delete-topic delete-topic})

(defn if-exists [if-exists]
  {:op :if-exists
   :children children/if-exists
   :if-exists if-exists})

(defn insert-columns [columns]
  {:op :insert-columns
   :children children/insert-columns
   :elements (mapv identifier columns)})

(defn insert [db table columns]
  {:op :insert
   :children children/insert
   :db db
   :table (identifier table)
   :columns (insert-columns columns)})

(defn source-name [source-name]
  {:op :source-name
   :children children/source-name
   :identifier (identifier source-name)})

(defn relation-primary [relation-primary]
  {:op :relation-primary
   :children children/relation-primary
   :source-name (source-name relation-primary)})

(defn aliased-relation [aliased-relation]
  {:op :aliased-relation
   :children children/aliased-relation
   :relation-primary (relation-primary aliased-relation)})

(defn relation [relation]
  {:op :relation
   :children children/relation
   :aliased-relation (aliased-relation relation)})

(defn from [from]
  {:op :from
   :children children/from
   :relation (relation from)})

(defn where [expr]
  {:op :where
   :children children/where
   :expression (expression expr)})

(defn order-by [columns]
  {:op :order-by
   :children children/order-by
   :columns columns})

(defn select-item [item]
  {:op :select-item
   :children children/select-item
   :expression
   (cond
     (and (map? item)
          (= :as (:op item)))
     item
     :else (expression item))})

(defn select-items [items]
  {:op :select-items
   :children children/select-items
   :items (mapv select-item items)})

(defn query [db selection]
  {:op :query
   :children children/query
   :db db
   :select-items (select-items selection)})

(defn show [db type]
  {:op :show
   :children children/show
   :db db
   :type type})

(defn print [db clause]
  {:op :print
   :children children/print
   :db db
   :clause clause})

(defn with [opts]
  {:op :with
   :children children/with
   :opts opts})

(defn table-property [table-element]
  {:op :table-property
   :children children/table-property
   :table-property (table-property table-element)})

(defn base-type [base-type]
  {:op :base-type
   :children children/base-type
   :type (identifier base-type)})

(defn- array-type? [type]
  (and (vector? type) (= 1 (count type))))

(defn array-type [[definition]]
  {:op :array-type
   :children children/array-type
   :definition definition})

(defn- map-type? [type]
  (and (map? type) (= 1 (count type))))

(defn map-type [definition]
  {:op :map-type
   :children children/map-type
   :definition definition})

(defn- struct-type? [type]
  (and (vector? type) (= :struct (first type))))

(defn struct-field [[name type]]
  {:op :struct-field
   :children children/struct-field
   :name name
   :type type})

(defn struct-type [[type fields]]
  {:op :struct-type
   :children children/struct-type
   :fields (mapv struct-field fields)})

(defn create-sink-connector [db name]
  {:op :create-sink-connector
   :children children/create-sink-connector
   :db db
   :name (identifier name)})

(defn create-source-connector [db name]
  {:op :create-source-connector
   :children children/create-source-connector
   :db db
   :name (identifier name)})

(defn table-element [[name type]]
  {:op :table-element
   :children children/table-element
   :identifier (identifier name)
   :type (cond
           (keyword? type)
           (base-type type)
           (array-type? type)
           (array-type type)
           (map-type? type)
           (map-type type)
           (struct-type? type)
           (struct-type type))})

(defn table-elements [elements]
  {:op :table-elements
   :children children/table-elements
   :elements (map table-element elements)})

(defn create-stream [db stream body]
  (cond-> {:op :create-stream
           :children children/create-stream
           :db db
           :source-name (source-name stream)}
    (sequential? (first body))
    (assoc :table-elements (table-elements (first body)))))

(defn create-table [db table body]
  (cond-> {:op :create-table
           :children children/create-table
           :db db
           :source-name (source-name table)}
    (sequential? (first body))
    (assoc :table-elements (table-elements (first body)))))

(defn drop-connector [db connector]
  {:op :drop-connector
   :children children/drop-connector
   :db db
   :connector (identifier connector)})

(defn drop-table [db table]
  {:op :drop-table
   :children children/drop-table
   :db db
   :table (identifier table)})

(defn drop-stream [db stream]
  {:op :drop-stream
   :children children/drop-stream
   :db db
   :stream (identifier stream)})

(defn group-by [columns]
  {:op :group-by
   :children children/group-by
   :expressions (mapv expression columns)})

(defn join-type [type]
  {:op :join-type
   :children children/join-type
   :type type})

(defn join [table condition type]
  {:op :join
   :children children/join
   :table table
   :condition (expression condition)
   :type (join-type type)})

(defn terminate [db query]
  {:op :terminate
   :children children/terminate
   :db db
   :query (identifier query)})

(defn value [columns value]
  {:op :value
   :children children/value
   :columns columns
   :value value})

(defn values [columns values]
  {:op :values
   :children children/values
   :columns columns
   :values (mapv #(value columns %) values)})

(defn window-expression [expression]
  {:op :window-expression
   :children children/window-expression
   :expression expression})

(defn window-type [type]
  {:op :window-type
   :children children/window-type
   :type type})

(defn window [type expressions]
  {:op :window
   :children children/window
   :type (window-type type)
   :expressions (mapv window-expression expressions)})
