(ns clojure.network.ip
  #?(:clj
      (:refer-clojure :exclude [first more cons count]))
  #?(:clj
      (:import
        [java.math.BigInteger]
        [clojure.lang ISeq Counted IPersistentSet]
        [java.net InetAddress Inet4Address Inet6Address]
        [java.lang UnsupportedOperationException])
    :cljs
     (:require [goog.net.IpAddress :as ip])))

(defprotocol IPConstructor
  (make-ip-address [this]))

(defprotocol IPInfo
  (ip-address [this])
  (version [this])
  (numeric-value [this]))

(defrecord IPAddress [value]
  IPInfo
  (ip-address [this]
    (str this))
  (version [this]
    #?(:clj
        (cond
          (instance? Inet4Address (InetAddress/getByAddress value)) "v4"
          (instance? Inet6Address (InetAddress/getByAddress value)) "v6")))
  (numeric-value [this]
    #?(:clj (BigInteger. value)))
  IPConstructor
  (make-ip-address [this] this)
  Object
  (toString [this]
    #?(:clj (.getHostAddress (InetAddress/getByAddress value)))))



(extend-type String
  IPConstructor
  (make-ip-address [this]
    (->IPAddress
      #?(:clj (.getAddress (InetAddress/getByName this))))))

#?(:clj
    (extend-type (Class/forName "[B")
      IPConstructor
      (make-ip-address [this]
        (->IPAddress this))))

#?(:clj
    (extend-type BigInteger
        IPConstructor
        (make-ip-address [this]
          (->IPAddress (.toByteArray this)))))

#?(:clj
    (extend-type clojure.lang.BigInt
      IPConstructor
      (make-ip-address [this]
        (->IPAddress (.toByteArray (biginteger this))))))

#?(:clj
    (defmethod print-method IPAddress [record writter]
        (.write writter
                (str "IP Address: " (str record) \newline
                     "Version: " (version record) \newline))))


(defn- get-network-address [ip subnet]
  (make-ip-address
    (reduce
      #?(:clj (fn [n bit] (.clearBit n bit)))
      (numeric-value ip)
      (case (version ip)
        "v4" (range (clojure.core/- 32 subnet))
        "v6" (range (range (clojure.core/- 128 subnet)))))))

(defn- get-broadcast-address [ip subnet]
  (make-ip-address
    (reduce
      #?(:clj (fn [n bit] (.setBit n bit)))
      (numeric-value ip)
      (case  (version ip)
        "v4" (range (clojure.core/- 32 subnet))
        "v6" (range (clojure.core/- 128 subnet))))))


(defn- get-all-addresses [ip subnet]
  (let [min-address (numeric-value (get-network-address ip subnet))
        max-address #?(:clj
                        (.add (numeric-value (get-broadcast-address ip subnet)) BigInteger/ONE)
                       :cljs nil)]
    (map make-ip-address (range min-address max-address))))


(def test-ipv6 "2a00:c31:1ffe:fff::9:13")
(def test-ipv4 "192.168.250.111")


#?(:clj
    (deftype Network [ip mask]
      IPInfo
      (ip-address [_] ip)
      (version [_] (version ip))
      (numeric-value [_] (numeric-value ip))
      ISeq
      (first [this] (get-network-address ip mask))
      (next [this] (next (get-all-addresses ip mask)))
      (more [this] (rest (get-all-addresses ip mask)))
      (count [this]
        (let [min-value (numeric-value (get-network-address ip mask))
              max-value (numeric-value (get-broadcast-address ip mask))]
          ;; TODO this function is not workin as meant to.
          ;; There is problem with returning value for whom
          ;; count coerces return value to long.
          ;; Point is no more thant 2^32 can be counted
          ;; BIG FAIL for... clojure
          #?(:clj (.subtract max-value min-value))))
      clojure.lang.Seqable
      (seq [this] (get-all-addresses ip mask))
      clojure.lang.IPersistentSet
      (disjoin [this _] #?(:clj (throw (Exception. "Network can't disjoin IP Addresses."))))
      (contains [this ip]
        (let [value (-> ip make-ip-address numeric-value)]
          (and (<= value (numeric-value (get-broadcast-address ip mask))) (>= value (numeric-value (get-network-address ip mask))))))
      (get [this address-number]
        #?(:clj
            (let [target-address (make-ip-address
                                   (.add
                                     (numeric-value (get-network-address ip mask))
                                     (biginteger address-number)))]
              (if (.contains this target-address)
                target-address
                (throw (IndexOutOfBoundsException. (str "IP address: " target-address " is not in network " this)))))))
      Object
      (toString [this] (str ip "/" mask))))


(defn make-network
  ([^String network] (apply make-network (clojure.string/split network #"/")))
  ([ip-address subnet]
   (when-let [ip-address (make-ip-address ip-address)]
     (case (version ip-address)
       "v4" (cond
              (string? subnet) (if-let [subnet (try (Integer/parseInt subnet) (catch Exception e nil))]
                                 (do
                                   (assert (and (<= subnet 32) (>= subnet 0)) (str "Subnet " subnet " is out of range."))
                                   (->Network ip-address subnet))
                                 (do
                                   (assert (seq (re-find #"\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}" subnet)) (str "Subnet: " subnet " is not valid!"))
                                   (let [subnet-count (fn subnet-count [x]
                                                        (let [testers (range 7 -1 -1)
                                                              test! (map #(bit-test x %) testers)
                                                              subnet-bits (count (take-while true? test!))
                                                              all-bits (count (filter true? test!))]
                                                          (if (not= subnet-bits all-bits)
                                                            (throw
                                                              #?(:clj (Exception. (str "Subnet part: " x " is not valid subnet!"))))
                                                            subnet-bits)))
                                         subnet (reduce clojure.core/+ (map subnet-count (map read-string (clojure.string/split subnet #"\."))))]
                                        (->Network ip-address subnet))))
              (number? subnet) (do
                                 (assert (and (<= subnet 32) (>= subnet 0)) (str "Subnet " subnet " is out of range."))
                                 (->Network ip-address subnet))
              :else #?(:clj (throw (UnsupportedOperationException. (str "Don't recongize subnet " (str subnet))))
                       :cljs nil))
       "v6" (cond
              (string? subnet) (if-let [subnet (try (Integer/parseInt subnet) (catch Exception e nil))]
                                 (->Network ip-address subnet)
                                 #(:clj (throw (Exception. (str "Can't make subnet from: " subnet)))))
              (number? subnet) (->Network ip-address subnet)
              :else #?(:clj (throw (UnsupportedOperationException. (str "Don't recongize subnet " (str subnet))))
                       :cljs nil))))))

#?(:clj
    (defmethod print-method Network [record writter]
      (.write writter
              (let [ip (.ip record)
                    subnet (.mask record)]
                (str "Network: " ip \/  subnet \newline
                     "Network address: " (get-network-address ip subnet) \newline
                     "Broadcast address: " (get-broadcast-address ip subnet) \newline
                     "Address count: " (count record)
                     \newline)))))


