(ns kolko.cloud-backend-database.models
  (:require [clojure.spec.alpha :as s]
            [clojure.string :refer [lower-case starts-with?]]
            [clojure.set :refer [rename-keys]]))

(s/def ::id (s/and int? pos?))
(s/def ::uuid uuid?)
(s/def ::seed-campaign-id ::id)
(s/def ::platform-id (s/or :id ::id :none nil?))
(s/def ::ad-text-id ::id)
(s/def ::ad-text-ids (s/and (s/coll-of ::ad-text-id) seq))

(s/def ::traffic-source #{"ms-ads" "google-ads"})
(s/def ::platform-status #{"active" "inactive" "expired" "paused" "deleted" "local-only"
                           "budget-and-manual-paused" "budget-paused" "suspended"})
(s/def ::editorial-status #{"active" "inactive" "active-limited" "under-review" "disapproved"
                            "eligible" "pending-review" "local-only"})
(s/def ::dki boolean?)
(s/def ::keyword (s/and string? #(> (count %) 0)))
(s/def ::keywords (s/coll-of ::keyword))
(s/def ::default-keyword ::keyword)
(s/def ::force-keyword (s/or :force ::keyword :dki nil?))
(s/def ::seed-keyword ::keyword)
(s/def ::tag int?)
(s/def ::route-tag ::tag)
(s/def ::country (s/and string? #(= (count %) 2) #(= (lower-case %) %)))
(s/def ::countries (s/coll-of ::country :min-count 1))
(s/def ::language (s/and string? #(= (count %) 2) #(= (lower-case %) %)))
(s/def ::languages (s/coll-of ::language :min-count 1))
(s/def ::url-base (s/and string? #(starts-with? % "https://")))
(s/def ::remote-ad-account (s/keys :req [::id ::url-base]))

(s/def ::budget (s/and float? pos?))
(s/def ::bid    (s/and float? pos?))
(s/def ::cpc    (s/and float? #(>= % 0.0)))
(s/def ::volume (s/and int?   #(>= % 0)))

(s/def ::overlap int?)
(s/def ::pct-overlap float?)
(s/def ::similarity-score (s/keys :req [::id ::overlap ::pct-overlap]))

(s/def ::route (s/keys :req [::tag ::id]))

(s/def ::ad-text (s/keys :req [::title1 ::title2 ::title3 ::text1 ::text2 ::url-path ::is-default
                               ::language ::country]
                         :opt [::id]))

(s/def ::ad       (s/keys :req [::force-keyword ::platform-status ::editorial-status ::route-tag ::ad-text-id
                                ::default-keyword]
                          :opt [::id ::ad-text ::platform-id]))
(s/def ::ads      (s/coll-of ::ad))

(s/def ::adgroup-keyword  (s/keys :req [::editorial-status ::platform-status ::keyword]
                                  :opt [::platform-id ::quality-score ::cpc ::keyword-id ::adgroup-id]))
(s/def ::adgroup-keywords (s/coll-of  ::adgroup-keyword))

(s/def ::adgroup  (s/keys :req [::bid ::ads ::platform-status ::adgroup-keywords]
                          :opt [::id ::platform-id]))
(s/def ::adgroups (s/coll-of ::adgroup))
(s/def ::campaign (s/keys :req [::traffic-source ::platform-status ::seed-campaign-id ::adgroups
                                ::countries ::languages ::budget ::remote-ad-account]
                          :opt [::id ::platform-id]))

(s/def ::campaign-overview (s/keys :req [::bids ::local-only-keywords ::active-keywords ::num-adgroups
                                         ::platform-id ::platform-status ::budget
                                         ::live-tags ::live-dkis ::live-forces
                                         ::id ::created ::modified ::traffic-source
                                         ::seed-campaign-id ::seed-campaign-keyword]))

(s/def ::research-keyword  (s/keys :req [::keyword ::cpc ::volume]))
(s/def ::research-keywords (s/coll-of ::research-keyword))

(s/def ::seed-campaign (s/keys :req [::seed-keyword ::country ::research-keywords]
                               :opt [::created ::id]))

(s/def ::similarity-score  (s/keys :req [::id ::overlap ::seed-keyword ::pct-overlap]))
(s/def ::similarity-scores (s/coll-of ::similarity-score))

(defn ad-text-json->clj [ad-text]
  (rename-keys ad-text
               {:title1 ::title1
                :title2 ::title2
                :title3 ::title3
                :text1  ::text1
                :text2  ::text2
                :url_path ::url-path
                :is_default ::is-default
                :language ::language
                :country ::country
                :id ::id}))

(defn ad-text-clj->json [ad-text]
  (rename-keys ad-text
               {::title1 :title1
                ::title2 :title2
                ::title3 :title3
                ::text1 :text1
                ::text2 :text2
                ::url-path :url_path
                ::is-default :is_default
                ::language :language
                ::country :country
                ::id :id}))

;;; ** Message Queue Models ;;;
(s/def ::msg (s/or :campaign-pause!   ::campaign-pause!
                   :campaign-unpause! ::campaign-unpause!
                   :create-local-only-campaigns!          ::create-local-only-campaigns!
                   :create-local-only-campaigns-from-ids! ::create-local-only-campaigns-from-ids!))
(s/def ::campaign-pause! (s/cat ::msg-name #(= % ::campaign-pause!)
                                ::ids (s/coll-of ::id :min-count 1)))

(s/def ::campaign-unpause! (s/cat ::msg-name #(= % ::campaign-unpause!)
                                  ::ids (s/coll-of ::id :min-count 1)))

(s/def ::create-local-only-campaigns! (s/cat ::msg-name #(= % ::create-local-only-campaigns!)))

(s/def ::create-local-only-campaigns-from-ids! (s/cat ::msg-name #(= % ::create-local-only-campaigns-from-ids!)
                                                      ::ids (s/coll-of ::id :min-count 1)))
