(ns web.authenticate
  (:require [connector.elastic :as elastic]
            [crypto.password.scrypt :as password]
            [clj-time.core :as t]
            [clj-time.coerce :as c]
            [digest])
  (:gen-class))

(defn- wayin [& pass]
  (apply str (into ["hYX8OiLCp1nlxlYn6tuUWNYveS32veiHdum7CVQlCbre3WRBeLI"] pass)))

(defn ->id [email]
  (digest/sha1 (wayin email)))

(defn ->password [pass]
  (password/encrypt (wayin pass)))

(defn ->token [& things-in]
  (digest/sha1 (apply str (into things-in 
                                [(c/to-long (t/now))]))))

(defn +mapping-uauth []
  (try (elastic/delete "uauth")
    (catch Exception e nil))
  (elastic/create-mapping "uauth"
                          {"user" {:properties
                                   {:email {:type "string"}
                                   :password {:type "string"}
                                   :token {:type "string"}
                                   :cgtoken {:type "string"}
                                   :balance {:type "float"}
                                   :confirm {:type "string"}}}}))

(defn exists?
  [email]
  (let [id (->id email)]
    (not (nil? (elastic/fetch (elastic/build-uri "uauth" "user" id))))))

(defn +mapping-life
  []
  (+mapping-uauth))
(defn user!
  ([id]
   (elastic/fetch (elastic/build-uri "uauth" "user" id)))
  ([email pass]
   (user! email pass nil))
  ([email pass cgtoken]
   (let [id (->id email)
         existing (elastic/fetch (elastic/build-uri "uauth" "user" id))]
     (if (nil? existing)
       (let [p (elastic/put (elastic/build-uri "uauth" "user" id)
                            {:email email
                             :pass (->password pass)
                             :token (->token email pass (c/to-long (t/now)))
                             :cgtoken cgtoken})]
         p)
       (if (password/check (wayin pass)
         (-> existing :_source :pass))
         existing)))))

(defn load-by-token
  [token]
  (let [existing (elastic/search-by-terms ["uauth"] ["user"]
                                 {:token token})]
    (-> existing :hits :hits first )))

(defn load-by-pass 
  [email pass & no-new-token]
  (if (exists? email)
    (let [existing (user! email pass)]
      (if-not (nil? existing)
        (if (first no-new-token)
          existing
          (let [new-token (->token email pass)
            _ (println "a new token" new-token)
              extra-update (if (or (= "" (-> existing :_source :cgtoken))
                                   (nil? (-> existing :_source :cgtoken)))
                             {:cgtoken (-> existing :_source :confirm)}
                             {})
              _ (elastic/update-doc "uauth" "user" (->id email) (merge
                                                                  {:token new-token}
                                                                  extra-update))]
          (load-by-pass email pass true)))))))



(defmacro with-token
  "takes a token, loads the authenticated user by the given token into the user-symbol.
  web.authenticate uses elasticsearch .. so you'll need one!"
  [token user-symbol & body]
  `(let [~user-symbol (web.authenticate/load-by-token ~token)]
     (when-not (nil? ~user-symbol)
       ~@body)))



