(ns mold.core
  (:require [clojure.spec.alpha :as spec]
            [spec-tools.core :as st]
            [spec-tools.data-spec :as ds]
            [datascript.core :as datascript]
            [mold.utils]
            #?(:cljs [re-posh.core]))
  (:import #?(:clj [java.util UUID])))


(def models-registry
  (atom #{}))


(def schemas-registry
  (atom {}))


(def spec-fns
  {::string    'cljs.core/string?
   ::number    'cljs.core/number?
   ::bool      'cljs.core/boolean?
   ::uuid      'cljs.core/uuid?
   ::keyword   'cljs.core/keyword?
   ::inst      'cljs.core/inst?
   ::map       'cljs.core/map?
   ::vector    'cljs.core/vector?
   ::not-empty '(cljs.core/complement cljs.core/empty?)})


(defn tag->attr [tag]
  (cond
    (= ::identity tag) [:db/unique :db.unique/identity]
    (= ::value tag) [:db/unique :db.unique/value]
    (= ::component tag) [:db/isComponent true]
    (= ::index tag) [:db/index true]
    (= ::ref tag) [:db/valueType :db.type/ref]
    (= ::many tag) [:db/cardinality :db.cardinality/many]
    (string? tag) [:db/doc tag]
    :else nil))


(defn read-schema [model schema singleton]
  (let [init-schema (if singleton
                      {(keyword model "singleton-id") {:db/unique :db.unique/identity}}
                      {})
        schema      (for [[attr-key tags] schema]
                      [(keyword model (name attr-key))
                       (into
                         {:db/cardinality :db.cardinality/one}
                         (if (vector? tags)
                           (map tag->attr tags)
                           (map tag->attr (vals tags))))])]
    (into init-schema schema)))


(defn ->spec-fns [tags]
  (let [fn-validators (->> (keys tags)
                           (map spec-fns)
                           (filter identity))]
    (cond-> fn-validators
            (and (= (::type tags) ::ref)
                 (= (::cardinality tags) ::many))
            (conj `(cljs.spec.alpha/coll-of ~(::ref-type tags)))

            (and (= (::type tags) ::ref)
                 (not (contains? tags ::cardinality)))
            (conj (::ref-type tags))

            (= (::type tags) ::enum)
            (conj `(cljs.spec.alpha/spec ~(::vals tags))))))


(defn ->map [tags]
  (if (vector? tags)
    (zipmap tags (repeat true))
    tags))


(defn ->spec-key [model [attr tags]]
  (let [tags-map      (->map tags)
        required?     (::req tags-map)
        nilable?      (::nilable tags-map)
        ref?          (= (::type tags-map) ::ref)
        ns-attr       (keyword model (name attr))
        spec-attr-key (if required? `(ds/req ~ns-attr) `(ds/opt ~ns-attr))
        spec-fns      (->spec-fns tags-map)
        spec-fns-val  (case (count spec-fns)
                        0 nil
                        1 (first spec-fns)
                        `(cljs.spec.alpha/and ~@spec-fns))]
    {:ref? ref?
     :item [spec-attr-key (if (and (some? spec-fns-val)
                                   nilable?)
                            `(ds/maybe ~spec-fns-val)
                            spec-fns-val)]}))


(defn defspecs [model attrs validations ref-mapping]
  (let [refs           (atom {})
        spec-items     (for [attr attrs
                             :let [{:keys [ref? item]} (->spec-key model attr)]
                             :when (some? (second item))]
                         (do
                           (when ref?
                             (swap! refs assoc (first attr) :id))
                           item))
        spec-map       (into {} spec-items)
        spec-name      (keyword (str *ns*) (str model ".root-spec"))
        core-spec-name (keyword "core.model" model)
        ref-mappings   (merge @refs ref-mapping)
        validators     (mapv (fn [{:keys [key message fn]}]
                               `(st/spec ~fn {:reason {~key ~message}}))
                             validations)]
    (if (empty? validators)
      `(st/spec {:name ~spec-name
                 :spec (with-meta (ds/spec {:name ~core-spec-name
                                            :spec ~spec-map})
                                  {:core.model/ref-mapping ~ref-mappings})})

      `(st/spec {:name ~spec-name
                 :spec (cljs.spec.alpha/and
                         (with-meta (ds/spec {:name ~core-spec-name
                                              :spec ~spec-map})
                                    {:core.model/ref-mapping ~ref-mappings})
                         ~@validators)}))))


(defn defevents [model singleton singleton-id]
  (let [add-name        (gensym (str "add-" model))
        add-many-name   (gensym (str "add-many-" model))
        update-name     (gensym (str "update-" model))
        delete-name     (gensym (str "delete-" model))
        singleton-ident [(keyword model "singleton-id") singleton-id]]
    [`(defn ~add-name [~'_ [model#]]
        [(cond-> {}
                 ~singleton (assoc ~(keyword model "singleton-id") ~singleton-id)
                 :always (merge (mold.utils/remove-nils-recursive model#)))])
     `(re-posh.core/reg-event-ds
        ~(keyword model "add")
        [re-frame.core/trim-v]
        ~add-name)

     `(defn ~add-many-name [~'_ [models#]]
        (mapcat #(~add-name nil [%]) models#))
     `(re-posh.core/reg-event-ds
        ~(keyword model "add-many")
        [re-frame.core/trim-v]
        ~add-many-name)

     (if singleton
       `(defn ~update-name [ds# [attrs# options#]]
          (let [eid# (datascript.core/entid ds# ~singleton-ident)]
            (cond-> []
                    (:replace options#)
                    (concat
                      (reduce-kv
                        (fn [acc# attr# value#]
                          (conj acc# [:db.fn/retractAttribute eid# attr#]))
                        []
                        attrs#))

                    :always (vec)
                    :always (conj (merge (mold.utils/remove-nils-recursive attrs#)
                                         {:db/id eid#})))))
       `(defn ~update-name [ds# [ident# attrs# options#]]
          (let [eid# (datascript.core/entid ds# ident#)]
            (cond-> []
                    (:replace options#)
                    (concat
                      (reduce-kv
                        (fn [acc# attr# value#]
                          (conj acc# [:db.fn/retractAttribute eid# attr#]))
                        []
                        attrs#))

                    :always (vec)
                    :always (conj (merge (mold.utils/remove-nils-recursive attrs#)
                                         {:db/id eid#}))))))

     `(re-posh.core/reg-event-ds
        ~(keyword model "update")
        [re-frame.core/trim-v]
        ~update-name)

     (if singleton
       `(defn ~delete-name [~'_ ~'_]
          [[:db/retractEntity ~singleton-ident]])
       `(defn ~delete-name [~'_ [ident#]]
          [[:db/retractEntity ident#]]))

     `(re-posh.core/reg-event-ds
        ~(keyword model "delete")
        [re-frame.core/trim-v]
        ~delete-name)]))


(defn get-default-pattern [model spec]
  (->> spec
       (filter (fn [[_ field-spec]]
                 (let [props (if (vector? field-spec)
                               field-spec
                               (vals field-spec))]
                   (> (.indexOf props ::component) -1))))
       keys
       (map (fn [prop-key]
              (keyword model (name prop-key))))
       (reduce (fn [acc prop-key]
                 (conj acc {prop-key ['*]}))
               ['*])))


(defn defsubs [model default-pattern singleton singleton-id uniq-attr]
  (let [singleton-ident [(keyword model "singleton-id") singleton-id]]
    [(if singleton
       `(re-posh.core/reg-sub
          ~(keyword model "prop")
          (fn [~'_ [~'_ attr#]]
            {:type      :query
             :query     '[:find ~'?value .
                          :in ~'$ ~'?e ~'?attr
                          :where [~'?e ~'?attr ~'?value]]
             :variables [~singleton-ident attr#]}))

       `(re-posh.core/reg-query-sub
          ~(keyword model "prop")
          '[:find ~'?value .
            :in ~'$ ~'?e ~'?attr
            :where [~'?e ~'?attr ~'?value]]))

     `(re-posh.core/reg-query-sub
        ~(keyword model "eids")
        '[:find [~'?eid ...]
          :where [~'?eid ~uniq-attr]])

     (if singleton
       `(re-posh.core/reg-sub
          ~(keyword model "pull")
          (fn [~'_ [~'_ pattern#]]
            {:type    :pull
             :pattern (or pattern# (quote ~default-pattern))
             :id      ~singleton-ident}))

       `(re-posh.core/reg-sub
          ~(keyword model "pull")
          (fn [~'_ [~'_ eid# pattern#]]
            {:type    :pull
             :pattern (or pattern# (quote ~default-pattern))
             :id      (if (map? eid#)
                        (:db/id eid#)
                        eid#)})))

     `(re-posh.core/reg-sub
        ~(keyword model "pull-many")
        (fn []
          (re-posh.core/subscribe [~(keyword model "eids")]))
        (fn [eids# [~'_ pattern#]]
          {:type    :pull-many
           :pattern (or pattern# (quote ~default-pattern))
           :ids     eids#}))]))


(defn find-uniq-attr [schema]
  (or
    ;; find unique attribute
    (reduce-kv
      (fn [_ k v]
        (when (some? (or (:db/unique v)
                         (:db/index v)))
          (reduced k)))
      nil
      schema)
    ;; or get the first one
    (ffirst schema)))


(defmacro defmodel [name spec & [{:keys [validations singleton ref-mapping]
                                  :or   {singleton false}}]]
  ;; TODO find models from classpath by meta attached to symbols and require those namespaces
  (let [model-name (str name)]
    (when (contains? @models-registry model-name)
      (println (format "Warning! (Skip in dev mode) Model %s has been already defined" model-name)))

    (let [schema                (read-schema model-name spec singleton)
          specs                 (defspecs model-name spec validations ref-mapping)
          singleton-id          (UUID/randomUUID)
          model-events          (defevents model-name singleton singleton-id)
          default-model-pattern (get-default-pattern model-name spec)
          uniq-attr             (find-uniq-attr schema)
          model-subs            (defsubs model-name default-model-pattern singleton singleton-id uniq-attr)]
      ;; Add model schema to the registry
      (swap! models-registry conj model-name)
      (swap! schemas-registry merge schema)

      `(do
         ;; Define the model spec
         (def ~name ~specs)

         ;; Define model CRUD
         ~@model-events
         ~@model-subs

         ;; Transformer for json values
         (defn ~(symbol (str "->" model-name)) [x#]
           (st/conform! ~name x# mold.utils/model-transformer))))))


(spec/def ::model-field-ident
  (spec/or :ident-keyword #{::string ::number ::bool ::uuid ::keyword ::inst ::map ::vector ::set
                            ::nilable ::not-empty ::req ::enum
                            ::identity ::value ::component ::index ::ref ::many}
           :boolean-flag boolean?
           :doc-string string?
           :enum-set set?
           :ref-symbol symbol?))


(spec/def ::model-field-long
  (spec/map-of keyword? ::model-field-ident))


(spec/def ::model-field-short
  (spec/coll-of ::model-field-ident :kind vector?))


(spec/def ::model-field-spec
  (spec/or :map ::model-field-long
           :vector ::model-field-short))


(spec/def ::model-spec
  (spec/map-of keyword? ::model-field-spec :min-count 1))


(spec/def ::validator
  (spec/keys :req-un [::key ::message ::fn]))


(spec/def ::validations
  (spec/coll-of ::validator :kind vector?))


(spec/def ::model-options
  (spec/keys :opt-un [::validations ::singleton ::ref-mapping]))


(spec/fdef defmodel
  :args (spec/cat :model-name symbol?
                  :model-spec ::model-spec
                  :model-options (spec/* ::model-options)))


(defmacro defschemas [sym]
  (let [s @schemas-registry]
    `(def ~sym ~s)))


(spec/fdef defschemas
  :args (spec/cat :schemas-var-name symbol?))
