(ns s3-beam.handler
  (:require [clojure.data.json :as json]
            [ring.util.codec :refer [base64-encode]]
            [clj-time.core :as t]
            [clj-time.format :as tf])
  (:import (javax.crypto Mac)
           (javax.crypto.spec SecretKeySpec)
           (java.text SimpleDateFormat)
           (java.util Date TimeZone)))

(defn now-plus [n]
  "Returns current time plus `n` minutes as string"
  (let [f (SimpleDateFormat. "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'")]
    (.setTimeZone f (TimeZone/getTimeZone "UTC" ))
    (.format f (Date. (+ (System/currentTimeMillis) (* n 60 1000))))))

(defn policy
  "Generate policy for upload of `key` with `mime-type` to be uploaded
  within optional `expiration-window` (defaults to 60)."
  ([bucket key mime-type]
     (policy bucket key mime-type 60))
  ([bucket key mime-type expiration-window]
     (ring.util.codec/base64-encode
      (.getBytes (json/write-str { "expiration" (now-plus expiration-window)
                                   "conditions" [{"bucket" bucket}
                                                 {"acl" "public-read"}
                                                 ["starts-with" "$Content-Type" mime-type]
                                                 ["starts-with" "$key" key]
                                                 {"success_action_status" "201"}]})
                 "UTF-8"))))

(defn hmac-sha256 [key string]
  "Returns signature of `string` with a given `key` using SHA-256 HMAC."
  (ring.util.codec/base64-encode
   (.doFinal (doto (javax.crypto.Mac/getInstance "HmacSHA256")
               (.init (javax.crypto.spec.SecretKeySpec. (.getByte key) "HmacSHA256")))
             (.getBytes string "UTF-8"))))

(def zone->endpoint
  "Mapping of AWS zones to S3 endpoints as documented here:
   http://docs.aws.amazon.com/general/latest/gr/rande.html#s3_region"
  {"us-east-1"      "s3"
   "us-west-1"      "s3-us-west-1"
   "us-west-2"      "s3-us-west-2"
   "eu-west-1"      "s3-eu-west-1"
   "eu-central-1"   "s3-eu-central-1"
   "ap-southeast-1" "s3-ap-southeast-1"
   "ap-southeast-2" "s3-ap-southeast-2"
   "ap-northeast-1" "s3-ap-northeast-1"
   "sa-east-1"      "s3-sa-east-1"})

(defn canonical-query-string
  "specs at http://docs.aws.amazon.com/general/latest/gr/sigv4-add-signature-to-request.html"
  [{:keys [action algorithm credentials signed-headers expires date]}]
  (str "Action=" action
       "&X-Amz-Algorithm=" algorithm
       "&X-Amz-Credential=" credentials
       "&X-Amz-Date=" date
       "&X-Amz-Expires=" expires
       "&X-Amz-SignedHeaders" signed-headers))

(defn sig-string [{:keys [action] :as sig-params}]
  (str "PUT" "\n" action "\n" (canonical-query-string sig-params) "\n"))

(defn sign-upload [{:keys [file-name mime-type]}
                   {:keys [bucket aws-zone aws-access-key aws-secret-key]}]
  (assert (zone->endpoint aws-zone) "No endpoint found for given AWS Zone")
  (let [p (policy bucket file-name mime-type)
        endpoint (zone->endpoint aws-zone)
        sig-action (str "https://" bucket "." endpoint ".amazonaws.com/")
        sig-credentials (str aws-access-key "%2F" endpoint "%2F" "s3" "%2F" "aws4_request")
        sig-algorithm "AWS4-HMAC-SHA256"
        sig-signed-headers "host"
        sig-expires 21599
        sig-date (tf/unparse (tf/formatters :basic-date-time-no-ms) (-> 6 t/hours t/from-now))
        ]
    {:Action sig-action
     :X-Amz-Signature (hmac-sha256 aws-secret-key (sig-string {:action sig-action :algorithm sig-algorithm :credentials sig-credentials :signed-headers sig-signed-headers :expires sig-expires :date sig-date}))
     :X-Amz-Credential sig-credentials
     :X-Amz-Algorithm sig-algorithm
     :X-Amz-SignedHeaders sig-signed-headers
     :X-Amz-Expires sig-signed-headers
     :X-Amz-Date sig-date}))

(defn s3-sign [bucket aws-zone access-key secret-key]
  (fn [request]
    {:status 200
     :body   (pr-str (sign-upload (:params request) {:bucket bucket
                                                     :aws-zone aws-zone
                                                     :aws-access-key access-key
                                                     :aws-secret-key secret-key}))}))
