(ns kixipipe.storage.redshift
  "Interface with AWS redshift"
  (:require [clj-time.format                      :as tf]
            [clojure.java.jdbc                    :as sql]
            [clojure.string                       :as str]
            [clojure.tools.logging                :as log]
            [kixipipe.storage.s3                  :as s3]
            [kixipipe.storage.redshift.management :as mgmt]
            [kixipipe.storage.redshift.report     :as report]
            [kixipipe.protocols                   :as kixi]
            [schema.core                          :as s]
            [com.stuartsierra.component           :as component]))

(def date-formatter (:basic-date tf/formatters))

(defn timestamp
  "Convert org.joda.time.DateTime to Redshift default datetime string"
  [time]
  (tf/unparse (tf/formatter "yyyy-MM-dd HH:mm:ss") time))

(def COPY_S3_TO_REDSHIFT_SQL
  "COPY %s FROM '%s' CREDENTIALS '%s' DELIMITER '%s' MAXERROR %d %s EMPTYASNULL REMOVEQUOTES TRUNCATECOLUMNS ACCEPTINVCHARS Compupdate ON")

(def UNLOAD_TO_REDSHIFT_SQL_PREFIX
  "UNLOAD('")

(def UNLOAD_TO_REDSHIFT_SQL_SUFFIX
  "') TO '%s' CREDENTIALS '%s' DELIMITER '%s' %s ADDQUOTES NULL AS '' ALLOWOVERWRITE")

(defn- build-s3-copy-prefix [s3-key-uri-base
                             {:keys [src-name feed-name feed-suffix] :as item}
                             date]
  (str/join "/" [s3-key-uri-base src-name  (if feed-suffix (str feed-name "-" feed-suffix) feed-name)
                 (tf/unparse date-formatter date)]))

(defn build-s3-report-url [{:keys [src-name feed-name] :as item}
                           s3-key-uri-base
                           report-name]
  (str/join "/" [s3-key-uri-base src-name feed-name report-name ""]))


(defrecord RedshiftSession [jdbc copy-date-fn format-table-name-fn table-meta view-meta]
  component/Lifecycle
    (start [this]
    (println "Starting RedshiftSession")
    this)
  (stop [this]
    (println "Stopping RedshiftSession")
    this))

(def ^:private Config {:jdbc {:subprotocol String
                              :subname     String
                              :user        String
                              :password    String}
                       :format-table-name-fn s/Any
                       (s/optional-key :copy-date-fn) s/Any
                       :table-meta s/Any
                       :view-meta s/Any})

(defn mk-session [{:keys [jdbc copy-date-fn format-table-name-fn table-meta view-meta]
                   :or {copy-date-fn identity} :as config}]
  (s/validate Config config)
  (->RedshiftSession jdbc copy-date-fn format-table-name-fn table-meta view-meta))

(defn create-table [redshift item]
  (mgmt/create-table redshift item))

(defn cleanup-if-failure [redshift table-name]
  (when-not (mgmt/has-rows? redshift table-name)
    (mgmt/drop-table-if-exists redshift table-name)))

(defn copy-to-redshift
  ([redshift item]
   (let [table (create-table redshift item)]
     (copy-to-redshift redshift item table)
     (cleanup-if-failure redshift table)))
  ([redshift item table-name]
   (let [{:keys [table-meta copy-date-fn format-table-name-fn]
          :or {copy-date-fn identity}} redshift
         {db-spec :jdbc} redshift
         item (copy-date-fn item)
         {:keys [src-name feed-name
                 date max-errors
                 encoding delimiter]
          :or   {max-errors 1 delimiter \tab}} item
         cols       (get table-meta feed-name)
         s3-config  (:storage redshift)
         aws-creds  (s3/build-aws-creds-arg-from-template s3-config "aws_access_key_id={access-key};aws_secret_access_key={secret-key}")
         encoding   (if encoding (name encoding) "")
         s3-prefix (build-s3-copy-prefix (s3/key-uri-base s3-config) item date)]

     (log/info "Executing " (format COPY_S3_TO_REDSHIFT_SQL table-name s3-prefix "..." delimiter max-errors encoding))
     (sql/execute! db-spec [(format COPY_S3_TO_REDSHIFT_SQL table-name s3-prefix aws-creds delimiter max-errors encoding)])

     (if (mgmt/has-rows? redshift table-name)
       (log/info "Completed copy to " table-name "successfully")
       (log/warn "No Rows inserted into " table-name)))))

(defn update-view [redshift item]
  (mgmt/update-view redshift item))

(defn build-aws-creds [storage]
  (s3/build-aws-creds-arg-from-template storage
   "aws_access_key_id={access-key};aws_secret_access_key={secret-key}"))

(defn feed-report [redshift item]
  (let [{:keys [feed-name date delimiter encoding]
         :or   {delimiter \tab}}  item
         item                      (assoc item :feed-suffix "report")
         storage                   (:storage redshift)
         aws-creds                 (build-aws-creds storage)
         s3-prefix                 (build-s3-copy-prefix (s3/key-uri-base storage) item date)
         encoding                  (if encoding (name encoding) "")]
    (let [table-names (mgmt/all-tables-seq redshift feed-name)
          sql (str UNLOAD_TO_REDSHIFT_SQL_PREFIX (str/replace (report/multi-table-dated-contents-report-sql table-names) "'" "''") UNLOAD_TO_REDSHIFT_SQL_SUFFIX)]
      (log/info "Executing " (format sql s3-prefix "..." delimiter encoding))
      (sql/execute! redshift
       (format sql s3-prefix aws-creds delimiter encoding)))))

(defn- quoted [s]
  (str \' s \'))

(defmulti ->db-quoted-string class )

(defmethod ->db-quoted-string org.joda.time.DateTime [dt]
  (quoted (timestamp dt)))

(defmethod ->db-quoted-string :default [s]
  (quoted s))


(defn sql-map-replace [sql m]
  (reduce-kv (fn [ret k v]
               (clojure.string/replace ret
                                       (re-pattern (str "(.*)" (str k) "(.*)"))
                                       (str "$1" (->db-quoted-string v) "$2")))
             sql m))

(defn copy-data [redshift from-date item]
  (let [{:keys [feed-name from-date date]} item
        {:keys [format-table-name-fn]}   redshift
        from-table-name                  (format-table-name-fn (assoc item :date from-date))
        table-name                       (format-table-name-fn item)]
    (mgmt/copy-data redshift from-table-name table-name)))

(defn execute-update [{db-spec :jdbc} table-name argmap]
  (let [update (sql-map-replace (format "UPDATE %s SET %s" table-name
                                        (apply str (->> argmap
                                                        (map (fn [[k v]] (str (name k) \= k)))
                                                        (interpose \,))))
                                argmap)]
    (log/info update)
    (sql/execute! db-spec [update])))

(defn execute-query [{db-spec :jdbc} sql argmap]
  (let [query (sql-map-replace sql argmap)]
    (log/info "Executing query:" query)
    (sql/query db-spec [query])))
