(ns kixipipe.pipeline
  "Short package description."
  (:require [clojure.tools.logging :as log]
            [pipejine.core         :as pipe]
            [clojure.stacktrace    :as st]
            [clojure.tools.macro   :as mac]
            [clojure.string        :as str]))

(defn submit-item [pipe job]
  (pipe/produce pipe job))

(defn shutdown-pipe [pipe]
  (pipe/shutdown pipe))

(defn pipe-produce-done [pipe]
  (pipe/produce-done pipe))

(defn produce-item [item & qs]
  (doseq [q qs]
    (log/debug "producing:" item "to " (:name q))
    (pipe/produce q item)))

(defn produce-items [items & qs]
  (doseq [q qs]
    (log/debugf "producing %d items to %s" (count items) (:name q))
    (log/debug "fitem:" (first items))

    (doseq [item items]
      (pipe/produce q item))))

(defmacro defnconsumer
  "Defines a fn that will consume the given queue"
  [q & fn-tail]
  (let [[fn-name [args & body]] (mac/name-with-attributes q fn-tail)
        fn# (symbol (str (name q) "-consumer"))]
    `(pipe/spawn-consumers
      ~q
      ;; This is somewhat convoluted and maybe simplifiable.
      ;; Goals for any rewrite.
      ;;   + Terse definition of pipeline with common exception handling 'built-in'
      ;;   + Logging of items when the enter each pipeline node/consumer.
      ;;   + Logging of the failing item close to the exception logging, so when error
      ;;     is emailed to support address, can work out what needs to be done ASAP.
      ;;   + Good stacktraces with few anonymous functions.
      ;;   + Log stacktraces over multiple lines to simplify reading and also
      ;;     to mitigate the length limitation of syslog
      (fn [item#]
        (log/info "Got " item#)
        (let [log-exception#
              (fn [t#]
                (let [stacktrace# (java.io.StringWriter.)]
                  (binding [*out* stacktrace#]
                    (println "Error processing " (pr-str item#))
                    (st/print-stack-trace t#)
                    (run! #(log/error %)
                          (str/split (str stacktrace#) #"\n")))))
              f# (fn ~fn# ~args
                   (let []
                     (try
                       ~@body
                       (catch java.sql.SQLException sqle#
                         (let [e# (or (.getNextException sqle#) sqle#)]
                           (log-exception# e#))
                         nil)
                       (catch Throwable t#
                         (log-exception# t#)
                         nil))))]
          (f# item#))))))
