(ns kixipipe.misc
  "Common functions that don't fit anywhere else."
  (:require [clojure.string        :as str]
            [kixipipe.ioplus       :as ioplus]
            [clj-time.core         :as t]
            [clj-time.format       :as tf]
            [clj-time.coerce       :as tc]
            [clojure.stacktrace    :as st]
            [clojure.tools.logging :as log])
  (:import [java.util Date]
           [java.io ByteArrayInputStream]))

(def ^{:private true} kixi-data-formatter (tf/formatter "yyyy_MM_dd"))
(def ^{:private true} timestamp-date-formatter (tf/formatter "yyyyMMddhhmmssSS"))
(def storage-date-formatter (:basic-date tf/formatters))
(defn same-day? [dt1 dt2]
  (let [match-attrs (juxt t/year t/month t/day)]
    (= (match-attrs dt1) (match-attrs dt2))))

(defn item-matches? [i1 i2]
  (and (= (:src-name i1) (:src-name i2))
       (= (:feed-name i1) (:feed-name i2))
       (same-day? (:date i1) (:date i2))
       (= (:checksum i1) (:checksum i2))))

(defn maybe-date [dt]
  (when dt
    (tf/unparse kixi-data-formatter dt)))

(defn maybe-timestamp [dt]
  (cond
   (nil? dt) (str (System/currentTimeMillis))
   (string? dt) dt
   (instance? Date dt) (when dt
                         (tf/unparse timestamp-date-formatter
                                     (tc/from-date dt)))))

(defn- extract-parts [{:keys [feed-name date
                              report-name
                              metadata filename
                              extension-override]}
                     timestamp?]
  (let [{:keys [timestamp hour part]}  metadata]
    (remove nil?
            [feed-name
             report-name
             (or hour (maybe-date date))
             (when timestamp? (maybe-timestamp timestamp))
             part])))

(defn- to-std-filename
  "takes a feed item and generates a local filename using the
   available attributes. For a given feed item this should always
   generate the same filename. Optionally supply a `timestamp?`
   flag to indicate whether the timestamp should be included in
   the filename. Typically you want to include the timestamp in
   temporary storage (e.g. a local download), but not in long
   term (versioned) storage.

   By default the filename extension of the original file is
   preserved, but extension override can be specified that will be used
   instead.

   Passing \"\" as the extension indicates no extension should be set."
  [item & [{:keys [timestamp?]}]]
  (let [parts           (extract-parts item timestamp?)
        [base ext]      (ioplus/base-and-exts (:filename item))
        final-extension (if-let [s (:extension-override item)]
                          (not-empty s)
                          ext)]
    (str (str/join "_" parts) (when final-extension (str "." final-extension)))))

(defn remote-filename-of
  "generate a filename suitable for remote storage"
  [item]
  (to-std-filename item {:timestamp? false}))

(defn local-filename-of
 "generate a filename suitable for local storage"
 [item]
  (to-std-filename item {:timestamp? true}))

(defn wrap-exception-handling [fn-name f]
  (let [log-exception
        (fn [e]
          (let [writer (java.io.StringWriter.)]
            (binding [*out* writer]
              (st/print-stack-trace e)
              (log/error (str fn-name ":" writer)))))]
    (fn [arg]
      (try
        (if (map? arg)
          (log/debugf "Got %s" (pr-str arg))
          (log/debugf "Got %s items, first: %s" (count arg) (pr-str (first arg))))
        (f arg)
        (catch java.sql.SQLException t
          (log-exception (or (.getNextException t) t))
          nil)
        (catch Throwable t
          (log/error arg)
          (log-exception t)
          nil)))))

(defn unparse-user-metadata-timestamp
  "Converts date to string in user metadata."
  [ts]
  (->> ts
       tc/from-date
       (tf/unparse (tf/formatter "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'"))))
