(ns kixipipe.transport.ftp
  "Short package description."
  (:require [kixipipe.protocols :as kixi]
            [clj-time.format    :as tf]
            [miner.ftp          :as ftp]
            [clojure.java.io    :as io]
            [potemkin])
  (:import [java.io IOException]
           [org.apache.commons.net.ftp FTP FTPClient FTPFile]))

(potemkin/import-vars [miner.ftp 
                       client-get-stream
                       client-complete-pending-command])

(defn ->type [n]
  (condp = n
    FTPFile/FILE_TYPE          :file
    FTPFile/DIRECTORY_TYPE     :directory
    FTPFile/SYMBOLIC_LINK_TYPE :symbolic-link
    :unknown))

(extend-protocol kixi/Mappable
  FTPFile
  (kixi/->map
    [file]
    (let [link? (.isSymbolicLink file)]
      (merge {:filename    (.getName file)
              :dir?        (.isDirectory file)
              :link?       link?
              :raw-listing (.getRawListing file)
              :size        (.getSize file)
              :timestamp   (.getTimestamp file)
              :type        (->type (.getType file))
              :user        (.getUser file)
              :group       (.getGroup file)}
             (if link?
               {:link (.getLink file)})))))

(defn- add-dir [dir m]
  (assoc m :dir dir))

(defn to-ftp-url [{:keys [user pass host dir]}]
  (io/as-url (str "ftp://" user ":" pass "@" host "/" dir)))

(defn ftp-file-seq [session dir]
  (let [client @(:ftp-client session)]
    (let [root {:dir "." :filename "/" :dir? true}
          branch? :dir?
          children (fn [{:keys [dir filename]}]
                     (let [resource (str dir "/" filename)]
                       (when (ftp/client-cd client resource)
                         (let [pwd (ftp/client-pwd client)]
                           (->> client
                                ftp/client-FTPFiles-all
                                (map kixi/->map)
                                (map (partial add-dir pwd)))))))]
        (doall (tree-seq branch? children root)))))

(defmacro with-session
  "Establish an FTP connection, bound to client, for the FTP url, and execute the body with
   access to that client connection.  Closes connection at end of body.  Keyword
   options can follow the url in the binding vector.  By default, uses a passive local data
   connection mode.  Use [client url :local-data-connection-mode :active] to override."
  [binding & body]
  `(let [~(binding 0) ~(binding 1)
         client#      (reset! (:ftp-client ~(binding 1)) (ftp/open (:url ~(binding 1))))
         u#           (:url ~(binding 0))]
     (when client#
       (try
         (when-let [user-info# (.getUserInfo u#)]
           (let [[^String uname# ^String pass#] (.split user-info# ":" 2)]
             (.login client# (ftp/decode uname#) (ftp/decode pass#))))
         (doto client#
           (.changeWorkingDirectory (.getPath u#))
           (.setFileType FTP/BINARY_FILE_TYPE)
           (.setControlKeepAliveTimeout 300)
           (.setFileTransferMode FTP/COMPRESSED_TRANSFER_MODE)
           (.enterLocalPassiveMode))
         ~@body
         (catch IOException e# (.getMessage e#) (throw e#))
         (finally (when (.isConnected client#)
                    (try
                      (.disconnect client#)
                      (catch IOException e2# (log/error e2# "Error disconnecting")))))))))
