(ns kixipipe.storage.s3.multipart-upload
  "Multipart upload to s3.
  Copied and pasted from https://github.com/weavejester/clj-aws-s3/blob/master/src/aws/sdk/s3.clj
  Modified to allow for providing metadata to multipart upload."
  (:require [clojure.walk          :as walk]
            [kixipipe.misc         :as misc]
            [clojure.tools.logging :as log])
  (:import com.amazonaws.services.s3.model.ObjectMetadata
           com.amazonaws.ClientConfiguration
           com.amazonaws.auth.BasicSessionCredentials
           com.amazonaws.auth.BasicAWSCredentials
           com.amazonaws.services.s3.AmazonS3Client
           com.amazonaws.services.s3.model.InitiateMultipartUploadRequest
           com.amazonaws.services.s3.model.AbortMultipartUploadRequest
           com.amazonaws.services.s3.model.CompleteMultipartUploadRequest
           com.amazonaws.services.s3.model.UploadPartRequest
           java.util.concurrent.Executors))
(defmacro set-attr
  "Set an attribute on an object if not nil."
  {:private true}
  [object setter value]
  `(if-let [v# ~value]
     (~setter ~object v#)))

(defn- map->ObjectMetadata
  "Convert a map of object metadata into a ObjectMetadata instance.
  Stringify user metadata (including date and timestamp).

  NOTE: Modified to remove :date and strigifiy :timestamp
  in user metadata."
  [metadata]
  (doto (ObjectMetadata.)
    (set-attr .setCacheControl         (:cache-control metadata))
    (set-attr .setContentDisposition   (:content-disposition metadata))
    (set-attr .setContentEncoding      (:content-encoding metadata))
    (set-attr .setContentLength        (:content-length metadata))
    (set-attr .setContentMD5           (:content-md5 metadata))
    (set-attr .setContentType          (:content-type metadata))
    (set-attr .setServerSideEncryption (:server-side-encryption metadata))
    (set-attr .setUserMetadata
              (-> metadata
                  (dissoc
                   :cache-control
                   :content-disposition
                   :content-encoding
                   :content-length
                   :content-md5
                   :content-type
                   :server-side-encryption
                   :date)
                  (update-in [:timestamp] misc/unparse-user-metadata-timestamp)
                  (walk/stringify-keys)))))

(defn- s3-client*
  [cred]
  (let [client-configuration (ClientConfiguration.)]
    (when-let [conn-timeout (:conn-timeout cred)]
      (.setConnectionTimeout client-configuration conn-timeout))
    (when-let [socket-timeout (:socket-timeout cred)]
      (.setSocketTimeout client-configuration socket-timeout))
    (when-let [max-retries (:max-retries cred)]
      (.setMaxErrorRetry client-configuration max-retries))
    (when-let [max-conns (:max-conns cred)]
      (.setMaxConnections client-configuration max-conns))
    (when-let [proxy-host (get-in cred [:proxy :host])]
      (.setProxyHost client-configuration proxy-host))
    (when-let [proxy-port (get-in cred [:proxy :port])]
      (.setProxyPort client-configuration proxy-port))
    (when-let [proxy-user (get-in cred [:proxy :user])]
      (.setProxyUsername client-configuration proxy-user))
    (when-let [proxy-pass (get-in cred [:proxy :password])]
      (.setProxyPassword client-configuration proxy-pass))
    (when-let [proxy-domain (get-in cred [:proxy :domain])]
      (.setProxyDomain client-configuration proxy-domain))
    (when-let [proxy-workstation (get-in cred [:proxy :workstation])]
      (.setProxyWorkstation client-configuration proxy-workstation))
    (let [aws-creds
          (if (:token cred)
            (BasicSessionCredentials. (:access-key cred) (:secret-key cred) (:token cred))
            (BasicAWSCredentials. (:access-key cred) (:secret-key cred)))

          client (AmazonS3Client. aws-creds client-configuration)]
      (when-let [endpoint (:endpoint cred)]
        (.setEndpoint client endpoint))
      client)))

(def ^{:private true :tag AmazonS3Client}
  s3-client
  (memoize s3-client*))

(defn- initiate-multipart-upload
  "If you want to provide any metadata describing the object being uploaded,
  you must provide it in the request to initiate multipart upload.

  NOTE: This has been modified by passing metadata to upload request:
  (-> (InitiateMultipartUploadRequest. bucket key (map->ObjectMetadata metadata)))"
  [cred bucket key metadata]
  (.getUploadId (.initiateMultipartUpload
                  (s3-client cred)
                  (-> (InitiateMultipartUploadRequest. bucket key (map->ObjectMetadata metadata))))))

(defn- abort-multipart-upload
  [{cred :cred bucket :bucket key :key upload-id :upload-id}]
  (.abortMultipartUpload
    (s3-client cred)
    (AbortMultipartUploadRequest. bucket key upload-id)))

(defn- complete-multipart-upload
  [{cred :cred bucket :bucket key :key upload-id :upload-id e-tags :e-tags}]
  (.completeMultipartUpload
    (s3-client cred)
    (CompleteMultipartUploadRequest. bucket key upload-id e-tags)))

(defn- upload-part
  [{cred :cred bucket :bucket key :key upload-id :upload-id
    part-size :part-size offset :offset ^java.io.File file :file}]
  (.getPartETag
   (.uploadPart
    (s3-client cred)
    (doto (UploadPartRequest.)
      (.setBucketName bucket)
      (.setKey key)
      (.setUploadId upload-id)
      (.setPartNumber (+ 1 (/ offset part-size)))
      (.setFileOffset offset)
      (.setPartSize ^long (min part-size (- (.length file) offset)))
      (.setFile file)))))

(defn put-multipart-object
  "Do a multipart upload of a file into a S3 bucket at the specified key.
  The value must be a java.io.File object.  The entire file is uploaded
  or not at all.  If an exception happens at any time the upload is aborted
  and the exception is rethrown. The size of the parts and the number of
  threads uploading the parts can be configured in the last argument as a
  map with the following keys:
    :part-size - the size in bytes of each part of the file.  Must be 5mb
                 or larger.  Defaults to 5mb
    :threads   - the number of threads that will upload parts concurrently.
                 Defaults to 16
    :metadata  - map of metadata that can include any of the
                 following keys:
                 :cache-control          - the cache-control header (see RFC 2616)
                 :content-disposition    - how the content should be downloaded by browsers
                 :content-encoding       - the encoding of the content (e.g. gzip)
                 :content-length         - the length of the content in bytes
                 :content-md5            - the MD5 sum of the content
                 :content-type           - the mime type of the content
                 :server-side-encryption - set to AES256 if SSE is required."
  [cred bucket key ^java.io.File file & [{:keys [part-size threads metadata]
                            :or {part-size (* 5 1024 1024) threads 16}}]]
  (let [upload-id (initiate-multipart-upload cred bucket key metadata)
        upload    {:upload-id upload-id :cred cred :bucket bucket :key key :file file}
        pool      (Executors/newFixedThreadPool threads)
        offsets   (range 0 (.length file) part-size)
        tasks     (map #(fn [] (upload-part (assoc upload :offset % :part-size part-size)))
                       offsets)]
    (try
      (complete-multipart-upload
        (assoc upload :e-tags (map #(.get ^java.util.concurrent.Future %)  (.invokeAll pool tasks))))
      (catch Exception ex
        (abort-multipart-upload upload)
        (throw ex))
      (finally (.shutdown pool)))))
