(ns kixipipe.storage.s3.status
  "Short package description."
  (:require [kixipipe.storage :as storage]
            [aws.sdk.s3        :as s3]
            [cheshire.core     :as json]
            [cheshire.generate :refer [add-encoder]]
            [clojure.java.io   :as io]
            [clojure.string    :as str]
            [clj-time.format   :as tf]
            [clj-time.coerce   :as tc]))

;; need to add this to store status report.
(add-encoder org.joda.time.DateTime
             (fn [^org.joda.time.DateTime dt ^com.fasterxml.jackson.core.JsonGenerator g]
               (.writeString g (tf/unparse (:date-time tf/formatters) dt))))
(add-encoder java.io.File
             (fn [^java.io.File f ^com.fasterxml.jackson.core.JsonGenerator g]
               (.writeString g (str f))))

(add-encoder java.util.GregorianCalendar
             (fn [^java.util.GregorianCalendar c ^com.fasterxml.jackson.core.JsonGenerator g]
               (.writeString g (tf/unparse (:date-time tf/formatters) (tc/from-long (.getTimeInMillis c))))))

(add-encoder java.lang.Character
             (fn [^java.lang.Character c ^com.fasterxml.jackson.core.JsonGenerator g]
               (.writeString g (str c))))

(defn- get-json-object-maybe [auth bucket key]
  (try
    (with-open [in (:content (s3/get-object auth bucket key))
                rdr (io/reader in)]
      (json/parse-stream rdr keyword))
    (catch com.amazonaws.AmazonServiceException e
      (if (= 404 (.getStatusCode e))
        nil
        (throw e)))))

(defn- put-json-object [auth bucket key obj]
  (s3/put-object auth bucket key (json/generate-string obj)))

(defn log-item-stored [session item]
  (let [{:keys [auth
                status-bucket]} session
        {:keys [src-name
                feed-name]}     item
        date-str                (storage/date-as-string :now)
        report-key              (str/join \/ [src-name feed-name date-str "status.json"])
        existing-status         (get-json-object-maybe auth status-bucket report-key)
        new-status              (update-in existing-status [(keyword src-name) (keyword feed-name)] conj item)]

    (put-json-object auth status-bucket report-key new-status)))
