(ns kixipipe.storage.redshift.report
  "Useful reports over redshift"
  (:require [clojure.java.jdbc :as sql]
            [clojure.string    :as str]))

(def ^{:private true} ALL_TABLE_REPORT_SQL
  "SELECT *
   FROM information_schema.tables
   WHERE table_schema = 'public'
   ORDER BY table_name")

(def ^{:private true} HOURS_CTE
  (str/join (interpose "\nUNION ALL\n" (map #(format "SELECT %d AS hour" %) (range 24)))))

(def ^{:private true} SELECT_DATE_HOUR_SQL
  "SELECT dh.date,
          dh.hour,
          (SELECT COUNT(*) FROM %<s t
             WHERE Date_Trunc('d',t.datetime) = dh.date AND
                   Date_Part('h',t.datetime) = dh.hour) as count
   FROM (SELECT d.date,h.hour FROM dates d CROSS JOIN hours h) dh
   ORDER  by 1,2\n")

(def ^{:private true} TABLE_BY_DATE_AND_HOUR_SQL
  (str
   "WITH hours AS (" HOURS_CTE "),\n"
   "dates AS (select distinct(Date_Trunc('d', datetime)) as date from %s)\n"
   SELECT_DATE_HOUR_SQL))

(defn table-report [{db-spec :jdbc}]
  (let [tables (sql/query db-spec
                          [ALL_TABLE_REPORT_SQL]
                          :row-fn (juxt :table_name :table_type))
        row-counts-sql (str/join
                        (interpose "\nUNION ALL\n"
                                   (map #(format "SELECT '%s' as table_name,
                                                          COUNT(*) AS row_count FROM %<s" %)
                                        (keep (fn [[n t]] (when (= "BASE TABLE" t) n))
                                              tables))))]
    (sql/query db-spec [row-counts-sql])))

(defn table-dated-contents-report [{db-spec :jdbc} table-name]
  (sql/query db-spec [(format TABLE_BY_DATE_AND_HOUR_SQL table-name)]))

(defn multi-table-dated-contents-report-sql [table-names]
  (apply str (interpose "\nUNION ALL\n"
                        (map #(format (str "(" TABLE_BY_DATE_AND_HOUR_SQL ")") %) table-names))))

(defn load-errors [{db-spec :jdbc} & {:keys [date feed-name]}]
  (sql/query db-spec ["SELECT * FROM loadview ORDER BY starttime DESC"]))
