(ns kixipipe.storage.redshift.management
  "Manage the database artifacts in redshift"
  (:require [clojure.core.match    :refer (match)]
            [clojure.java.io       :as io]
            [clojure.java.jdbc     :as sql]
            [clojure.set           :as set]
            [clojure.string        :as str]
            [clojure.tools.logging :as log]
            [schema.core           :as s]
            [clj-time.core         :as t]
            [clj-time.format       :as tf]
            [kixipipe.window       :as win]))

(def ^:private  SHOW_RELATIONS_SQL
  ;; NOTE: USE of ILIKE for case-insensitive matching. (Postgres extension)
  "SELECT table_name, table_type
     FROM information_schema.tables
     WHERE table_schema = 'public' And
           table_name ilike '%s' AND
           table_type = '%s'
     ORDER BY table_name DESC")

(def ^:private date-formatter (:basic-date tf/formatters))

(defn- table-exists? [{db-spec :jdbc} tname]
  (let [sql (format SHOW_RELATIONS_SQL tname "BASE TABLE")]
    (log/info sql)
    (first (sql/query db-spec [sql]))))

(defn- view-exists? [{db-spec :jdbc} vname]
  (let [sql (format SHOW_RELATIONS_SQL vname "VIEW")]
    (log/info sql)
    (first (sql/query db-spec [sql]))))

(defn all-tables-seq
  ([redshift vname]
     (all-tables-seq redshift vname identity))
  ([{db-spec :jdbc} vname filter-fn]
     (let [pattern (re-pattern (str "(?i)" vname "_\\d+"))]
       (->> (sql/query db-spec
                       [(format SHOW_RELATIONS_SQL (str vname "%") "BASE TABLE")]
                       :row-fn :table_name)
            ;; need to filter more than the SQL for segment_ / segment_feed_
            (filter (comp filter-fn (partial re-matches pattern)))))))

(defn- create-table-if-not-exists [{db-spec :jdbc :as redshift} tname colspecs]
  (when-not (table-exists? redshift tname)
    (let [sql (apply sql/create-table-ddl tname colspecs)]
      (log/info sql)
      (sql/execute! db-spec [sql]))))

(defn drop-table-if-exists [{db-spec :jdbc :as redshift} tname]
  (when (table-exists? redshift tname)
    (let [drop (format "DROP TABLE %s" tname)]
      (log/info drop)
      (sql/execute! db-spec [drop]))))

(defn- truncate-table [{db-spec :jdbc :as redshift} tname]
  (when (table-exists? redshift tname)
    (let [truncate (format "TRUNCATE TABLE %s" tname) ]
      (log/info truncate)
      (sql/execute! db-spec [truncate]))))

(defn truncate-table-if-exists [redshift tname]
  (when (table-exists? redshift tname)
    (truncate-table redshift tname)))

(defn- grant-select-to [{db-spec :jdbc} name & groups]
  (let [grant (format "GRANT SELECT ON %s TO %s" name (str/join \, (map (partial str "GROUP ") groups)))]
    (log/info grant)
    (sql/execute! db-spec [grant])))

(defn select-into [{db-spec :jdbc} new-name name]
  (let [select (format "SELECT * INTO %s FROM %s" new-name name)]
    (log/info select)
    (sql/execute! db-spec [select])))

(defn has-rows? [{db-spec :jdbc} tname]
  (pos? (count (sql/query db-spec [(str "SELECT COUNT(*) from " tname " as count")]))))

(defn- colspecs-from [redshift {:keys [feed-name]}]
  (get-in redshift [:table-meta (.toLowerCase feed-name)] []))

(defn- viewspecs-from [redshift {:keys [feed-name]}]
  (get-in redshift [:view-meta (.toLowerCase feed-name)] (get-in redshift [:view-meta :default])))

(defn create-table
  "create the feed table"
  [redshift item]
  (let [{:keys [feed-name]} item
        colspecs                 (colspecs-from redshift item)
        table-name               ((:format-table-name-fn redshift) item)]
    (log/info "Creating Table " table-name  " for feed " feed-name)
    (truncate-table-if-exists redshift table-name)
    (create-table-if-not-exists redshift  table-name colspecs)
    (grant-select-to redshift table-name "datascientist" "operator" "monitor")
    table-name))

(defn copy-data [redshift from-table-name table-name]
  (log/info "Creating Table " table-name  " by copying from " from-table-name)
  (drop-table-if-exists redshift table-name)
  (select-into redshift table-name from-table-name)
  (grant-select-to redshift table-name "datascientist" "operator"))

(def WindowDefinition (s/either (s/enum :today :yesterday :now :last-week :last-fortnight)
                                [(s/one (s/enum :month :start-of-month :day :hour) :key) s/Int]))

(def UnionView {:type (s/eq :union)})
(def LatestView {:type (s/eq :latest)})
(def SlidingView {:type (s/eq :sliding)
                  :window (s/either s/Keyword
                                    [(s/one WindowDefinition :start)
                                     (s/one WindowDefinition :end)])
                  (s/optional-key :suffix) String})

(def ViewConfig
  (s/conditional #(= (:type %) :union)  UnionView
                 #(= (:type %) :latest) LatestView
                 #(= (:type %) :sliding) SlidingView))

(def ViewsConfig
  [ViewConfig])

(defmulti create-view (fn [view redshift  _] (:type view)))

(defmethod create-view :latest [view {db-spec :jdbc :as redshift} item]
  (let [{:keys [feed-name view-filter-fn] :or {view-filter-fn identity}} item
        [latest & _]  (all-tables-seq redshift feed-name view-filter-fn)]

    (log/info "Creating view with spec " view)

     (when latest
      (let [sql (format "CREATE OR REPLACE VIEW %s AS SELECT * FROM %s" feed-name latest)]
        (log/info sql)
        (sql/execute! db-spec [sql])
        (log/info "Created view with spec " view)
        feed-name))))

(defmethod create-view :union [view {db-spec :jdbc :as redshift} item]
  (let [{:keys [feed-name view-filter-fn] :or {view-filter-fn identity}} item
        tables (all-tables-seq redshift feed-name view-filter-fn)]

    (log/info "Creating view with spec " view)

    (when (seq tables)
      (let [sql (apply str "CREATE OR REPLACE VIEW " feed-name " AS "
                       (interpose " UNION ALL "
                                  (mapv #(str "SELECT * FROM " %) tables)))]
        (log/info sql)
        (sql/execute! db-spec [sql])
        (log/info "Created view with spec " view)
        feed-name))))

(defmethod create-view :sliding [view redshift item]
  (let [{:keys [feed-name
                view-filter-fn]
         :or {view-filter-fn identity}} item
         {:keys [suffix
                 window
                 enforce-tables-exist]}  view
         {:keys [format-table-name-fn]}  redshift
         db-spec                         (:jdbc redshift)
         tables                          (into #{} (all-tables-seq redshift feed-name view-filter-fn))
         required-vnames                 (->> (win/dates-in-window window)
                                              (map #(hash-map :feed-name feed-name :date %1))
                                              (map format-table-name-fn)
                                              (filter view-filter-fn)
                                              (into (sorted-set)))
         feed-name                       (if suffix (str feed-name "_" suffix) feed-name)
         tables-in-view                  (set/intersection required-vnames tables)
         sql                             (apply str "CREATE OR REPLACE VIEW " feed-name " AS "
                                                (interpose " UNION ALL "
                                                           (mapv #(str "SELECT * FROM " %) tables-in-view)))]

    (log/info "Creating view with spec " view)

    (when-not (set/subset? required-vnames tables-in-view)
      (let [missing-required (set/difference required-vnames tables-in-view)]
        (if enforce-tables-exist
          (throw (ex-info "The following required tables do not exist!" {:missing missing-required}))
          (log/error (format "The following required tables do not exist! %s" missing-required)))))

    (log/info sql)
    (sql/execute! db-spec [sql])
    (log/info "Created view with spec " view)
    feed-name))

(defn update-view [redshift item]
  (let [view (viewspecs-from redshift item)]
    (s/validate ViewsConfig view) ;; TODO should be done at startup.

    (let [views (if (vector? view) view (vector view))]
      (doseq [view views]
        (let [viewname (create-view view redshift item)]
         (grant-select-to redshift viewname "datascientist" "operator"))))))
