(ns kixipipe.storage.gcs
  (:require [clj-time.coerce :as tc]
            [clojure.java.io :as io]
            [clojure.set :as set]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [kixipipe.google.auth :as google-auth]
            [kixipipe.ioplus :as ioplus]
            [kixipipe.protocols :as kixi]
            [schema.core :as s])
  (:import [com.google.api.services.storage Storage$Builder StorageScopes]
           org.apache.commons.codec.binary.Base64))

(def ^:private Config {:email-address String
                       :p12-file (s/pred ioplus/exists-as-file? "<credentials file>")
                       :bucket-name String
                       :download-dir (s/pred ioplus/exists-as-dir? "<a directory>")})

(defn- ->checksum [s]
  "Converts a Base64 encoded md5Hash into a hex string representation"
  (->> s
      Base64/decodeBase64
      (java.math.BigInteger. 1) ;; the '1' is the sign, means interpret as a positive number.
      (format "%x")))

(defn- build-storage-service [{:keys [credential]}]
  (-> (Storage$Builder. google-auth/HTTP_TRANSPORT
                        google-auth/JSON_FACTORY
                        credential)
      (.setApplicationName "kixi data feeder")
      (.build)
      (.objects)))

(defn- gcs-item->item [{:strs [bucket id md5Hash size updated name]} options]
  (merge {:bucket bucket
          :id id
          :name name
          :checksum (->checksum md5Hash)
          :metadata {:timestamp (tc/from-long (.getValue updated))}}
         options))

(defn- paged-results-seq [cmd options]
  (let [{:strs [items nextPageToken]} (.execute cmd)
        items (map #(gcs-item->item % options) items)]
    (if nextPageToken
      (lazy-cat
       items
       (paged-results-seq (doto cmd
                            (.setPageToken nextPageToken))
                          options))
      items)))


(defn- filename-from-item [{:keys [name metadata]}]
  (let [{:keys [timestamp]} metadata
        [base ext] (ioplus/base-and-ext name)]
    (format "%s_%s.%s" base (.getMillis timestamp) ext)))

(defn get-object [{:keys [download-dir bucket-name credential storage-objects]} item]
  (let [get  (doto (.get storage-objects bucket-name (:name item))
               (.setOauthToken (.getAccessToken credential)))
        filename (filename-from-item item)]
    (with-open [in (.executeMediaAsInputStream get)]
      (io/copy in (io/file download-dir filename)))
    (assoc item
           :dir download-dir
           :filename filename)))

(defn feed-name->prefix [feed-name]
  ;; TODO - validate feed-name.
  feed-name)

(defrecord GCSSession []
  component/Lifecycle
  (start [this]
    (println "Starting GCSSession")
    (let [credential      (google-auth/build-credential this #{StorageScopes/DEVSTORAGE_FULL_CONTROL})
          storage-service (build-storage-service credential)]
      (assoc this
             :credential credential
             :storage-objects (build-storage-service credential))))
  (stop [this]
    (println "Stopping GCSSession")
    this)
  kixi/StorageSession
  (list-items [{:keys [bucket-name credential storage-objects]} src-name feed-name options]
    (let [access-token (.getAccessToken credential)
          cmd          (doto (.list storage-objects bucket-name)
                         (.setPrefix (feed-name->prefix feed-name))
                         (.setOauthToken access-token))]
      (paged-results-seq cmd (merge options {:src-name src-name
                                             :feed-name feed-name})))))

(defn mk-session
  "Creates an google cloud storage session with the given config. This
  session should be passed to all google cloud storage service
  calls." [config]
   (s/validate Config config)
   (map->GCSSession config))

(comment

  (def config {:email-address "128906095177-etsb0pp20a3i9i8hd9oh2nktlha3h1ec@developer.gserviceaccount.com"
               :p12-file "/home/neale/Downloads/doubleclick-publishers-feed-da3f4a57bf24.p12"
               :bucket-name "gdfp-5765"
               :download-dir "/tmp"})

    (def sess (component/start (mk-session config)))
    (kixi/list-items sess nil "NetworkActiveViews" {})
    (kixi/list-items sess nil "bar" {})

    ;; {"bucket" "gdfp-5765", "contentType" "application/octet-stream", "crc32c" "a4iodQ==", "etag" "CMjnp/TyoccCEAI=", "generation" 1439325926061000, "id" "gdfp-5765/NetworkActiveViews_273272_20150811_10.gz/1439325926061000", "kind" "storage#object", "md5Hash" "us0zXOOO7NKZu/4vEqp/DA==", "mediaLink" "https://www.googleapis.com/download/storage/v1/b/gdfp-5765/o/NetworkActiveViews_273272_20150811_10.gz?generation=1439325926061000&alt=media", "metageneration" 2, "name" "NetworkActiveViews_273272_20150811_10.gz", "selfLink" "https://www.googleapis.com/storage/v1/b/gdfp-5765/o/NetworkActiveViews_273272_20150811_10.gz", "size" 144163756, "storageClass" "STANDARD", "updated" #<DateTime 2015-08-11T20:45:26.057Z>}
    )
