(ns kixipipe.ioplus
  "extensions to ."
  (:require [clojure.java.io :as io])
  (:import [java.io File FilterInputStream ByteArrayInputStream]
           [java.util.zip GZIPInputStream GZIPOutputStream]))

(defn mkdirs [^File d]
  (.mkdirs d))

(defn gzip-input-stream [s]
  (GZIPInputStream. (io/input-stream s)))

(defn gzip-output-stream [s]
  (GZIPOutputStream. (io/output-stream s)))

(defn string-input-stream
  ([s] (string-input-stream s "UTF-8"))
  ([s cs] (ByteArrayInputStream.
           (.getBytes s cs))))

(defprotocol BaseAndExtension
  (base-and-ext [x]
    "For an `x` representing a file /foo/bar/baz/base.ext1.ext2
     return [\"base.ext1\" \"ext2\"")
  (base-and-exts [x]
    "For an `x` representing a file /foo/bar/baz/base.ext1.ext2
     return [\"base\" \"ext1.ext2\""))

(extend-protocol BaseAndExtension
  nil
  (base-and-ext [_] nil)
  (base-and-exts [_] nil)  
  File
  (base-and-ext [x]
    (base-and-ext ^String (.getCanonicalPath x)))
  (base-and-exts [x]
    (base-and-exts ^String (.getCanonicalPath x)))
  String
  (base-and-ext [x]
    (rest (re-matches #"(?:.*/)?([\w-\.]+)\.(\w+)" x)))
  (base-and-exts [x]
    (rest (re-matches #"(?:.*/)?([\w-]+)\.((?:\w+\.*)+)" x))))


(defn append-suffix
  "takes a filename of the form /foo/bar/base.ext and returns /foo/bar/base-suffix.ext"
  [filename suffix]
  (let [[base exts] (base-and-exts filename)]
    (str base "-" suffix "." exts)))

(defn with-ext
  "takes a filename of the form /foo/bar/base.ext1.ext2 and returns /foo/bar/base.ext"
  [ext filename]
  (let [[base exts] (base-and-exts filename)]
    (str base ".tsv")))

(defn exists-as-file? [f]
  (let [f (io/as-file f)]
    (and (.exists f)
         (.isFile f))))

(defn exists-as-dir? [f]
  (let [f (io/as-file f)]
    (and (.exists f)
         (.isDirectory f))))

(defn mk-temp-file! [prefix suffix]
  (java.io.File/createTempFile prefix suffix))

(defn delete! [f]
  (when (.exists f)
    (.delete f)))
