(ns kixipipe.digest
  "Utils for handling Message digests on streams"
  (:require  [clojure.java.io       :as io]
             [clojure.tools.logging :as log])
  (:import [java.io ByteArrayInputStream InputStream]
           [java.security DigestInputStream DigestOutputStream MessageDigest]))

(defn- md5-bytes->string [md5]
    (apply str (map (partial format "%02x") md5)))

(defprotocol MD5Checksummable
  (md5-checksum [o])
  (md5-checksum-as-string [o]))

(extend-protocol MD5Checksummable
  DigestInputStream
  (md5-checksum [o]
    (.. o getMessageDigest digest))
  (md5-checksum-as-string [o]
    (md5-bytes->string (md5-checksum o)))
  DigestOutputStream
  (md5-checksum [o]
    (.. o getMessageDigest digest))
  (md5-checksum-as-string [o]
    (md5-bytes->string (md5-checksum o)))
  MessageDigest
  (md5-checksum [o]
    (.digest o))
    (md5-checksum-as-string [o]
    (md5-bytes->string (md5-checksum o))))

(defn md5-matches?
  "Do the bytes consumed by the DigestInputStream dis have a digest matching md5-str?"
  [dis md5-str]
  (when (and dis md5-str)
    (let [dis-md5 (md5-checksum-as-string dis)]
      (log/debugf "MD5 comparing %s to %s" dis-md5 md5-str)
      (= dis-md5 (.toLowerCase md5-str)))))

(defn md5-input-stream
  "Create a MD5 Digesting Input Stream reading from s"
  [s]
  (when s
    (DigestInputStream. (io/input-stream s) (MessageDigest/getInstance "MD5"))))

(defn md5-output-stream
  "Create a MD5 Digesting Output Stream wrting to s"
  [s]
  (when s
    (DigestOutputStream. (io/output-stream s) (MessageDigest/getInstance "MD5"))))

(defn- correct-checksum? [f item]
  (let [st   (-> f io/input-stream md5-input-stream)
        size (.available st)
        ba   (byte-array size)]
    (.read st ba 0 size)
    (md5-matches? st (:checksum item))))

(defn- copy-and-validate-checksum [src destdir destfile item]
  (with-open [src (md5-input-stream src)]
    (let [dest (io/file destdir destfile)
          checksum (:checksum item)]
      (log/debugf "download of %s %s" dest :started)
      (io/copy src dest)
      (if (md5-matches? src checksum)
        (log/debugf "download of %s %s" dest :complete)
        (throw (ex-info  "Checksum FAILED!" {:type ::bad-checksum :data item})))
      (assoc item :dir destdir :filename destfile))))

(defn- copy-and-calculate-checksum [src destdir destfile item]
  (with-open [src (md5-input-stream src)]
    (let [dest (io/file destdir destfile)]
      (log/debugf "copy of %s %s" dest :started)
      (io/copy src dest)
      (log/debugf "copy of %s %s" dest :complete)
      (assoc item :dir destdir :filename destfile :checksum (md5-checksum-as-string src)))))

(defn should-download? [dest item validate-checksum? download-existing?]
  (log/debug "should download " item ", val:" validate-checksum? ", down:" download-existing?)
  (let [file-exists? (.exists dest)]
    (if file-exists?
      (when download-existing?
        (and validate-checksum? (not (correct-checksum? dest item))))
      (not file-exists?))))

(defn read-md5-from-md5-file [dir filename]
  (let [md5-filename (io/file dir (str filename ".md5"))
        file-contents (when (.exists md5-filename) (slurp (io/file dir (str filename ".md5"))))
        [_ md5-str name] (re-matches #"(\p{XDigit}+)\s+(.+)" (or file-contents ""))]
    (assert md5-str (str "No MD5 found in file " md5-filename))))

(defn- write-md5-to-md5-file! [dir filename checksum]
  ;; standard is TWO spaces between checksum and filename.
  (spit (io/file dir (str filename ".md5")) (str checksum "  " filename \n)))

(defn copy-stream!
  "copies the src to the destination, validating checksums. src can be a delay object to allow lazy loading'"
  [src destdir destfile item & [options]]
  (let [{:keys [validate-checksum? download-existing?]
         :or {validate-checksum? true
              download-existing? true}} options
        dest                            (io/file destdir destfile)]
    (try
      (if (should-download? dest item validate-checksum? download-existing?)
        (let [item (if (:checksum item)
                     (copy-and-validate-checksum (force src) destdir destfile item)
                     (copy-and-calculate-checksum (force src) destdir destfile item))]
          (write-md5-to-md5-file! destdir destfile (:checksum item))
          item)
        (assoc item :dir destdir :filename destfile))
      (catch Throwable t
        ;; must delete dest if incomplete copy.
        (when (.exists dest)
          (.delete dest))
        (throw t) ;; will trigger logging.
        ))))
