(ns kixipipe.storage.redshift.report
  "Useful reports over redshift"
  (:require [clojure.java.jdbc.deprecated :as sql]
            [clojure.string               :as str]))

(def ^{:private true} ALL_TABLE_REPORT_SQL
  "SELECT *
   FROM information_schema.tables
   WHERE table_schema = 'public'
   ORDER BY table_name")

(def ^{:private true} HOURS_CTE
  (str/join (interpose "\nUNION ALL\n" (map #(format "SELECT %d AS hour" %) (range 24)))))

(def ^{:private true} SELECT_DATE_HOUR_SQL
  "SELECT dh.date,
          dh.hour,
          (SELECT COUNT(*) FROM %<s t
             WHERE Date_Trunc('d',t.datetime) = dh.date AND
                   Date_Part('h',t.datetime) = dh.hour) as count
   FROM (SELECT d.date,h.hour FROM dates d CROSS JOIN hours h) dh
   ORDER  by 1,2\n")

(def ^{:private true} TABLE_BY_DATE_AND_HOUR_SQL
  (str
   "WITH hours AS (" HOURS_CTE "),\n"
   "dates AS (select distinct(Date_Trunc('d', datetime)) as date from %s)\n"
   SELECT_DATE_HOUR_SQL))

(defn table-report [db]
  (sql/with-connection db
    (let [tables  (sql/with-query-results rows
                    [ALL_TABLE_REPORT_SQL]
                    (mapv (juxt :table_name :table_type) rows))
          row-counts-sql (str/join
                          (interpose "\nUNION ALL\n"
                                     (map #(format "SELECT '%s' as table_name,
                                                          COUNT(*) AS row_count FROM %<s" %)
                                          (keep (fn [[n t]] (when (= "BASE TABLE" t) n))
                                                tables))))]
      (sql/with-query-results rows
        [row-counts-sql] (doall rows)))))

(defn table-dated-contents-report [db table-name]
  (sql/with-connection db
    (sql/with-query-results rows
      [(format TABLE_BY_DATE_AND_HOUR_SQL table-name)]
      (doall rows))))

(defn multi-table-dated-contents-report-sql [table-names]
  (apply str (interpose "\nUNION ALL\n"
                        (map #(format (str "(" TABLE_BY_DATE_AND_HOUR_SQL ")") %) table-names))))

(defn load-errors [db & {:keys [date feed-name]}]
  (sql/with-connection db
    (sql/with-query-results rows
      ["SELECT *
        FROM loadview
        ORDER BY starttime DESC"]
      (doall rows))))
