(ns kixipipe.storage.redshift
  "Interface with AWS redshift"
  (:require [clj-time.format                      :as tf]
            [clojure.java.jdbc.deprecated         :as sql]
            [clojure.string                       :as str]
            [clojure.tools.logging                :as log]
            [kixipipe.storage.s3                  :as s3]
            [kixipipe.storage.redshift.management :as mgmt]
            [kixipipe.storage.redshift.report     :as report]
            [kixipipe.protocols                   :as kixi]
            [schema.core                          :as s]
            [com.stuartsierra.component           :as component]))

(def date-formatter (:basic-date tf/formatters))

(defn timestamp
  "Convert org.joda.time.DateTime to Redshift default datetime string"
  [time]
  (tf/unparse (tf/formatter "yyyy-MM-dd HH:mm:ss") time))

(def COPY_S3_TO_REDSHIFT_SQL
  "COPY %s FROM '%s' CREDENTIALS '%s' DELIMITER '%s' MAXERROR %d %s EMPTYASNULL REMOVEQUOTES TRUNCATECOLUMNS ACCEPTINVCHARS Compupdate ON")

(def UNLOAD_TO_REDSHIFT_SQL_PREFIX
  "UNLOAD('")

(def UNLOAD_TO_REDSHIFT_SQL_SUFFIX
  "') TO '%s' CREDENTIALS '%s' DELIMITER '%s' %s ADDQUOTES NULL AS '' ALLOWOVERWRITE")

(defn- build-s3-copy-prefix [s3-key-uri-base
                             {:keys [src-name feed-name feed-suffix] :as item}
                             date]
  (str/join "/" [s3-key-uri-base src-name  (if feed-suffix (str feed-name "-" feed-suffix) feed-name)
                 (tf/unparse date-formatter date)]))

(defn build-s3-report-url [{:keys [src-name feed-name] :as item}
                           s3-key-uri-base
                           report-name]
  (str/join "/" [s3-key-uri-base src-name feed-name report-name ""]))


(defrecord RedshiftSession [jdbc copy-date-fn format-table-name-fn table-meta view-meta]
  component/Lifecycle
    (start [this]
    (println "Starting RedshiftSession")
    this)
  (stop [this]
    (println "Stopping RedshiftSession")
    this))

(def ^:private Config {:jdbc {:subprotocol String
                              :subname     String
                              :user        String
                              :password    String}
                       :format-table-name-fn s/Any
                       (s/optional-key :copy-date-fn) s/Any
                       :table-meta s/Any
                       :view-meta s/Any})

(defn mk-session [{:keys [jdbc copy-date-fn format-table-name-fn table-meta view-meta]
                   :or {copy-date-fn identity} :as config}]
  (s/validate Config config)
  (->RedshiftSession jdbc copy-date-fn format-table-name-fn table-meta view-meta))

(defn copy-to-redshift [session item]
  (let [{:keys [jdbc table-meta copy-date-fn format-table-name-fn]
         :or {copy-date-fn identity}} session
        item   (copy-date-fn item)
        {:keys [src-name feed-name
                date max-errors
                encoding delimiter]
         :or   {max-errors 1 delimiter \tab}}  item
         cols                                  (get table-meta feed-name)
         s3-config                             (:storage session)
         aws-creds                             (s3/build-aws-creds-arg-from-template s3-config "aws_access_key_id={access-key};aws_secret_access_key={secret-key}")
         s3-prefix                             (build-s3-copy-prefix (s3/key-uri-base s3-config) item date)
         encoding                              (if encoding (name encoding) "")]

    (sql/with-connection jdbc
      (let [table-name (mgmt/create-table jdbc (format-table-name-fn item) table-meta item)]
        (log/info "Executing " (format COPY_S3_TO_REDSHIFT_SQL table-name s3-prefix "..." delimiter max-errors encoding))
        (sql/do-commands
         (format COPY_S3_TO_REDSHIFT_SQL table-name s3-prefix aws-creds delimiter max-errors encoding))
                (when-not (mgmt/has-rows? table-name)
          (log/warn "No Rows inserted into " table-name)
          (mgmt/drop-table-if-exists table-name))
        (log/info "Completed copy to " table-name "successfully")))))

(defn update-view [session item]
  (let [{:keys [jdbc view-meta]} session]
    (mgmt/update-view jdbc view-meta item)))

(defn build-aws-creds [storage]
  (s3/build-aws-creds-arg-from-template storage
   "aws_access_key_id={access-key};aws_secret_access_key={secret-key}"))

(defn feed-report [session item]
  (let [{:keys [feed-name date delimiter encoding]
         :or   {delimiter \tab}}  item
         {:keys [jdbc table-meta]} session
         item                      (assoc item :feed-suffix "report")
         storage                   (:storage session)
         aws-creds                 (build-aws-creds storage)
         s3-prefix                 (build-s3-copy-prefix (s3/key-uri-base storage) item date)
         encoding                  (if encoding (name encoding) "")]
    (sql/with-connection jdbc
      (let [table-names (mgmt/all-tables-seq feed-name)
            sql (str UNLOAD_TO_REDSHIFT_SQL_PREFIX (str/replace (report/multi-table-dated-contents-report-sql table-names) "'" "''") UNLOAD_TO_REDSHIFT_SQL_SUFFIX)]
        (log/info "Executing " (format sql s3-prefix "..." delimiter encoding))
        (sql/do-commands
         (format sql s3-prefix aws-creds delimiter encoding))))))

(defn- quoted [s]
  (str \' s \'))

(defmulti ->db-quoted-string class )

(defmethod ->db-quoted-string org.joda.time.DateTime [dt]
  (quoted (timestamp dt)))

(defmethod ->db-quoted-string :default [s]
  (quoted s))

(defn sql-map-replace [sql m]
  (reduce-kv (fn [ret k v]
               (clojure.string/replace ret
                                       (re-pattern (str "(.*)" (str k) "(.*)"))
                                       (str "$1" (->db-quoted-string v) "$2")))
             sql m))

(defn copy-data [session from-date item]
  (let [{:keys [feed-name to-date date]} item
        {:keys [format-table-name-fn]}   session
        from-table-name                  (format-table-name-fn item)
        table-name                       (format-table-name-fn (-> item (dissoc :date) (assoc :date to-date)))]
    (mgmt/copy-data session from-table-name table-name)))

(defn execute-query [session sql argmap]
  (let [query (sql-map-replace sql argmap)]
    (log/info "Executing query")
    (sql/with-connection (:jdbc session) (sql/do-prepared query))))
