(ns kixipipe.window
  (:require [clojure.core.match :refer (match)]
            [clj-time.core :as t]
            [clj-time.periodic :as p]))
(defn- current-day []
  (apply t/date-time ((juxt t/year t/month t/day) (t/now))))

(defn- current-hour []
  (apply t/date-time ((juxt t/year t/month t/day t/hour) (t/now))))

(defn before-or-same? [this that]
  (not (t/after? this that)))

;; periods - these shouldn't make sense without an offset. i.e. day
;; before yesterday is [:day -2] rather than [:yesterday -1]
(defmulti period->offset (fn [period _] period))
(defmethod period->offset :start-of-month [_ offset]
  (let [m (period->offset :month offset)]
    (t/date-time (t/year m) (t/month m))))
(defmethod period->offset :month [_ offset]
  (t/plus (current-day) (t/months offset)))
(defmethod period->offset :day [_ offset]
  (t/plus (current-day) (t/days offset)))
(defmethod period->offset :hour [_ offset]
  (t/plus (current-hour) (t/hours offset)))

;; keyword ids - these should be meaningful in their own right, you
;; wouldn't apply an offset to one of these
(defmulti keyword->offset identity)
(defmethod keyword->offset :today [_] (current-day))
(defmethod keyword->offset :yesterday [_] (t/plus (current-day) (t/days -1)))
(defmethod keyword->offset :now [_] (t/now))
(defmethod keyword->offset :last-week [_] (period->offset :day -7))
(defmethod keyword->offset :last-fortnight [_] (period->offset :day -14))

(defn dates-in-window
  "Return all the dates in the range defined by start s and end e (**inclusive**)"
  [[s e]]
  (let [[start end]  (match [s e]
                            [[p o] [p' o']] [(period->offset p o) (period->offset p' o')]
                            [[p o] k]       [(period->offset p o) (keyword->offset k)]
                            [k [p o]]       [(keyword->offset k) (period->offset p o)])]
    (take-while #(before-or-same? % end) (p/periodic-seq start (t/days 1)))))

(defn hours-in-window [[s e]]
  "Return all the hours in the range defined by start s and end e (**inclusive**)"
  (let [[start end]  (match [s e]
                            [[p o] [p' o']] [(period->offset p o) (period->offset p' o')]
                            [[p o] k]       [(period->offset p o) (keyword->offset k)]
                            [k [p o]]       [(keyword->offset k) (period->offset p o)])]
    (take-while #(before-or-same? % end) (p/periodic-seq start (t/hours 1)))))
