(ns kixipipe.storage.redshift.management
  "Manage the database artifacts in redshift"
  (:require [clojure.core.match     :refer (match)]
            [clojure.java.io        :as io]
            [clojure.java.jdbc      :as sql]
            [clojure.set            :as set]
            [clojure.string         :as str]
            [clojure.tools.logging  :as log]
            [schema.core            :as s]
            [clj-time.core          :as t]
            [clj-time.format        :as tf]
            [kixipipe.window        :as win]))

(def ^{:private true} SHOW_RELATIONS_SQL
  ;; NOTE: USE of ILIKE for case-insensitive matching. (Postgres extension)
  "SELECT table_name, table_type
     FROM information_schema.tables
     WHERE table_schema = 'public' And
           table_name ilike ? AND
           table_type = ?
     ORDER BY table_name DESC")

(def date-formatter (:basic-date tf/formatters))

(defn- table-exists? [tname]
  (sql/with-query-results rows [SHOW_RELATIONS_SQL tname "BASE TABLE"]
    (doall rows)))

(defn- view-exists? [vname]
  (sql/with-query-results rows [SHOW_RELATIONS_SQL vname "VIEW"]
    (doall rows)))

(defn all-tables-seq [vname]
  (sql/with-query-results rows
    [SHOW_RELATIONS_SQL (str vname "%") "BASE TABLE"]
    (let [pattern (re-pattern (str "(?i)" vname "_\\d+"))
          data (mapv :table_name rows)]
      (filter (partial re-matches pattern) data)))) ;; need to filter more than the SQL for segment_ / segment_feed_

(defn- create-table-if-not-exists [tname colspecs]
  (when-not (table-exists? tname)
    (log/debug (apply sql/create-table-ddl tname colspecs))
    (apply sql/create-table tname colspecs)))

(defn drop-table-if-exists [tname]
  (when (table-exists? tname)
    (sql/drop-table tname)))

(defn- truncate-table [tname]
  (sql/do-commands
   (str "TRUNCATE TABLE " tname)))

(defn truncate-table-if-exists [tname]
  (when (table-exists? tname)
    (truncate-table tname)))

(comment  ;;; DANGER WILL ROBINSON
          ;;; This will destroy your database!
  (defn drop-all-tables []
   (let [views (sql/with-query-results rows
                 [SHOW_RELATIONS_SQL "%" "VIEW"]
                 (mapv :table_name rows))
         tables (sql/with-query-results rows
                  [SHOW_RELATIONS_SQL "%" "BASE TABLE"]
                  (mapv :table_name rows))]
     (doseq [view views]
       (sql/do-commands (str "DROP VIEW " view)))
     (doseq [table tables]
       (sql/drop-table table)))))

(defn- grant-select-to [name & groups]
  (sql/do-commands
   (format "GRANT SELECT ON %s TO %s"
           name
           (str/join \, (map (partial str "GROUP ") groups)))))

(defn- format-table-name [name date]
  (format "%s_%s" name   (tf/unparse date-formatter date)))

(defn has-rows? [tname]
  (sql/with-query-results rows [(str "SELECT COUNT(*) from " tname " as count")]
    (pos? (:count (first rows)))))

(defn create-table
  "create the feed table"
  [db table-meta item]
  (let [{:keys [feed-name
                date]}    item
        table-name        (format-table-name feed-name date)
        colspecs          (get table-meta (.toLowerCase feed-name) [])]
    (log/info "Creating Table " table-name  "for feed " feed-name)
    (sql/with-connection db
      (truncate-table-if-exists table-name)
      (create-table-if-not-exists table-name colspecs)
      (grant-select-to table-name "datascientist" "operator"))
    table-name))


(def WindowDefinition (s/either s/Keyword
                                [(s/one (s/enum :month :start-of-month) :key) (s/one s/Any :value)]))

(def UnionView {:type (s/eq :union)})
(def LatestView {:type (s/eq :latest)})
(def SlidingView {:type (s/eq :sliding)
                  :window (s/either s/Keyword
                                    [(s/one WindowDefinition :start)
                                     (s/one WindowDefinition :end)])
                  (s/optional-key :suffix) String})

(def ViewConfig
  (s/conditional #(= (:type %) :union)  UnionView
                 #(= (:type %) :latest) LatestView
                 #(= (:type %) :sliding) SlidingView))

(def ViewsConfig
  [ViewConfig])

(defmulti create-view (fn [view _] (:type view)))

(defmethod create-view :latest [_ {vname :feed-name}]
  (let [[latest & _]  (all-tables-seq vname)]
     (when latest
      (let [sql (str "CREATE OR REPLACE VIEW " vname " AS SELECT * FROM " latest)]
        (log/debug sql)
        (sql/do-commands sql)
        vname))))

(defmethod create-view :union [_ {vname :feed-name}]
  (let [tables  (all-tables-seq vname)]
    (when (seq tables)
      (let [sql (apply str "CREATE OR REPLACE VIEW " vname " AS "
                       (interpose " UNION ALL "
                                  (mapv #(str "SELECT * FROM " %) tables)))]
        (log/debug sql)
        (sql/do-commands sql)
        vname))))

(defmethod create-view :sliding [view {vname :feed-name}]
  (let [{:keys [suffix window]} view
        tables                  (all-tables-seq vname)
        required-vnames         (->> (win/dates-in-window window)
                                     (map (partial format-table-name vname))
                                     (into (sorted-set)))
        vname                   (if suffix (str vname "_" suffix) vname)]

    (if-let [missing (seq (set/difference required-vnames (into #{} tables)))]
      (throw (Throwable. (format "The following required tables do not exist! %s" missing)))
      (let [sql (apply str "CREATE OR REPLACE VIEW " vname " AS "
                       (interpose " UNION ALL "
                                  (mapv #(str "SELECT * FROM " %) required-vnames)))]
        (log/debug sql)
        (sql/do-commands sql)
        vname))))

(defn update-view [db {:keys [view] :or {view [{:type :latest}]} :as item}]

  (s/validate ViewsConfig view) ;; TODO should be done at startup.

  (let [views (if (vector? view) view (vector view))]
    (doseq [view views]
      (sql/with-connection db
        (-> (create-view view item)
            (grant-select-to "datascientist" "operator"))))))
