(ns kixipipe.transport.sftp
  "Handle sftp connections to lotame"
  (:require [kixipipe.misc              :as misc]
            [kixipipe.digest            :as digest]
            [kixipipe.protocols         :as proto]
            [kixipipe.ioplus            :as io]
            [clj-ssh.ssh                :as ssh]
            [clojure.tools.logging      :as log]
            [schema.core                :as s]
            [com.stuartsierra.component :as component]
            [potemkin]
            )
  (:import [com.jcraft.jsch ChannelSftp ChannelSftp$LsEntry SftpATTRS SftpException]
           [java.util Date]
           [java.io ByteArrayInputStream ByteArrayOutputStream File]))

(potemkin/import-vars [clj-ssh.ssh connected? connect disconnect])

(def ^:dynamic *ssh-session*)

(defn- ->flags
  [flags]
  (let [mapping {SftpATTRS/SSH_FILEXFER_ATTR_ACMODTIME   :has-atime-mtime
                 SftpATTRS/SSH_FILEXFER_ATTR_EXTENDED    :has-extended
                 SftpATTRS/SSH_FILEXFER_ATTR_PERMISSIONS :has-permissions
                 SftpATTRS/SSH_FILEXFER_ATTR_SIZE        :has-size
                 SftpATTRS/SSH_FILEXFER_ATTR_UIDGID      :has-uid-gid}]
    (reduce (fn [a [k v]] (if (pos? (bit-and flags k)) (conj a v) a) ) #{} mapping)))

(extend-protocol proto/Mappable
  ChannelSftp$LsEntry
  (proto/->map [entry]
    {:filename (.getFilename entry)
     :attrs    (proto/->map (.getAttrs entry))})
  SftpATTRS
    (proto/->map [attrs]
      (let [flags (->flags (.getFlags attrs))]
        (merge
         {:dir?            (.isDir attrs)
          :link?           (.isLink attrs)
          :flags           flags}
         (when (flags :has-atime-mtime) {:atime       (Date. (* (.getATime attrs) 1000))
                                         :mtime       (Date. (* (.getMTime attrs) 1000))})
         (when (flags :has-extended)    {:extended    (seq (.getExtended attrs))})
         (when (flags :has-permissions) {:permissions (.getPermissions attrs)})
         (when (flags :has-size)        {:size        (.getSize attrs)})
         (when (flags :has-uid-gid))    {:uid         (.getUId attrs)
                                         :gid         (.getGId attrs)}))))

(defn mk-ssh-session
  "Create, but don't open an ssh session using the supplied config."
  [config]
  (let [agent       (ssh/ssh-agent {:use-system-ssh-agent false} )
        private-key (:ssh-key config)
        public-key  (str private-key ".pub")]
    (ssh/add-identity agent {:private-key-path private-key
                             :public-key-path public-key})
    (ssh/session agent (:host config) {:username (:user config)
                                       :strict-host-key-checking false})))

(defn- append-dir
  "Takes a map, as returned from an sftp ls command and updates the dir attribute"
  [dir {parent :dir :as m}]
  (assoc m :dir (str parent (when parent "/") dir)))

(defn- sftp-ls-maybe
  "ls a directory, if the directory doesn't exist return empty list"
  [channel dir]
  (try
    (ssh/sftp channel {} :ls dir)
    (catch SftpException e
      (if (= (.-id e) (ChannelSftp/SSH_FX_NO_SUCH_FILE))
        ()
        (throw e)))))

(defn sftp-file-seq
  "Do a depth first walk of the remote directory tree, starting at dir.
   Hidden files are skipped unless `:show-hidden? true` option is passed."
  [session dir & [options]]
  (let [{:keys [show-hidden?]} options
        channel  (ssh/sftp-channel *ssh-session*)
        root     {:filename dir :attrs {:dir? true}}
        branch?  #(-> % :attrs :dir?)
        hidden? (if show-hidden?
                  #(->> % :filename #{"." ".."}) ; must always skip these or recurse forever
                  #(->> % :filename (re-matches #"^\..*$")))
        children (fn [{:keys [filename]}]
                   (->> filename
                        (sftp-ls-maybe channel)
                        (map proto/->map)
                        (map (partial append-dir filename))
                        (remove hidden?)))]
    (ssh/with-channel-connection channel
      (doall (tree-seq branch? children root)))))

(defn sftp-get-to-file [session item]
  (let [{:keys [dir filename checksum]} item
        {:keys [download-dir]}          session
        channel                         (ssh/sftp-channel *ssh-session*)
        resource                        (str dir "/" filename)
        local-filename                  (misc/local-filename-of item)]
    (log/debugf "sftp-get-to-file: Downloading %s -> %s" resource local-filename)
     (ssh/with-channel-connection channel
       (digest/copy-stream! (delay  (ssh/sftp channel {} :get resource))
                           download-dir local-filename
                           item))))

(defn sftp-get-as-string
  "Downloads the sftp resource `src` and returns a string of the contents.
   WARNING: don't use for large files."
  [session src & [options]]
      (log/debugf "sftp-get-as-string: Downloading %s" src)
  (let [channel (ssh/sftp-channel *ssh-session*)]
    (ssh/with-channel-connection channel
      (slurp (ssh/sftp channel {} :get src)))))

(defn sftp-ls
  "Do a remote ls on a given directory"
  ([session]
     (sftp-ls session nil))
  ([session dir]
     (ssh/with-connection session
       (doall (map proto/->map (ssh/sftp session {} :ls dir))))))

(defrecord SFTPSession [config]
  component/Lifecycle
  (start [this]
    (println "Starting sftp session for " (:host config))
    this)
  (stop [this]
    (println "Stopping sftp session for " (:host config))
    this))

(defn mk-session [config]
  (SFTPSession. config))

; copied from clojure.core
(defmacro ^{:private true} assert-args
  [& pairs]
  `(do (when-not ~(first pairs)
         (throw (IllegalArgumentException.
                  (str (first ~'&form) " requires " ~(second pairs) " in " ~'*ns* ":" (:line (meta ~'&form))))))
     ~(let [more (nnext pairs)]
        (when more
          (list* `assert-args more)))))

(defmacro with-session
  "Using a session created by `mk-session` execute the body and
   close the session. Note - session cannot be reused after closing.

   Example usage:
   ```
      (with-session [session (mk-session my-config)]
          (sftp-file-seq session \".\"))
   ```"
   [binding & body]
   (assert-args
    (vector? binding) "a vector for binding"
    (= 2 (count binding)) "exactly 2 forms in binding vector")
   `(let [~(binding 0) ~(binding 1)]
      (binding [*ssh-session* (mk-ssh-session ~(binding 1))]
        (try
          (when-not (ssh/connected? *ssh-session*)
            (ssh/connect *ssh-session*))
          ~@body
          (finally
            (ssh/disconnect *ssh-session*))))))

(comment
  (def config    {:host "booth.crwdcntrl.net"
                  :user "anmedia"
                  :ssh-key "/Users/neale/.ssh/anmedia-mastodonc"
                  :download-dir "/tmp"})

  (with-session [session (mk-session config)]
    (ls session "20130410")
    (sftp-file-seq session ".")
    (sftp-get-as-string session "20130410/metadata.json.md5"))
)
