(ns kixipipe.scheduler
  "Scheduling functions"
  (:require [kixipipe.pipeline                    :refer [submit-item]]
            [kixipipe.protocols                   :as kixi]
            [clojure.tools.logging                :as log]
            [clojure.walk                         :as walk]
            [clojurewerkz.quartzite.conversion    :as qc]
            [clojurewerkz.quartzite.jobs          :as j]
            [clojurewerkz.quartzite.scheduler     :as qs]
            [clojurewerkz.quartzite.schedule.cron :as cron]
            [clojurewerkz.quartzite.triggers      :as trig]
            [schema.core                          :as s]
            [com.stuartsierra.component           :as component]))

(def ^{:private true} QUARTZ_OPTIONS
  {"org.quartz.scheduler.skipUpdateCheck" true
   "org.quartz.threadPool.threadCount"    1})

(j/defjob SubmitToPipeJob
  [ctx]
  (let [item (walk/keywordize-keys (qc/from-job-data ctx))
        pipeline (:pipeline item)
        item (dissoc item :pipeline)]
    (log/info "Submitting " item)
    (submit-item pipeline item)))

(defn build-job [pipeline item id]
    (j/build
     (j/of-type SubmitToPipeJob)
     (j/using-job-data (walk/stringify-keys (assoc item :pipeline (:head pipeline))))
     (j/with-identity (j/key (format "jobs.process-jobs.%02d" id)))))

(defn build-trigger [cron-str id]
   (trig/build
    (trig/with-identity (trig/key (format "triggers.%02d" id)))
    (trig/start-now)
    (trig/with-schedule (cron/schedule
                      (cron/cron-schedule cron-str)))))

(defn configure-quartz [opts]
  (doseq [[key value] (merge QUARTZ_OPTIONS opts)]
    (System/setProperty key (String/valueOf value))))

(defn configure-scheduler [config pipeline]
  (let [schedule             (:schedule config)
        process-job-schedule (:process-job-schedule schedule)]
    (qs/initialize)
    (dorun (map-indexed (fn [id [cron-str item]]
                          (qs/schedule (build-job pipeline item id)
                                       (build-trigger cron-str id)))
                        process-job-schedule))))

(defrecord KixiSchedulerSession [config]
  component/Lifecycle
  (start [this]
    (println "Starting Scheduler.")
    (configure-quartz (:quartz config))
    (configure-scheduler config (:pipeline this))
    (qs/start)
    (assoc this :scheduler @qs/*scheduler*))
  (stop [this]
    (println "Stopping Scheduler.")
    (qs/shutdown)
    (dissoc this :scheduler)))

(def ^:private Config {:process-job-schedule {String {s/Keyword s/Any}}
                       :quartz {String s/Any}})

(defn mk-session [config]
  (KixiSchedulerSession. config))

(defn pause-scheduler []
  (qs/standby))

(defn unpause-scheduler []
  (qs/start))
