(ns kixipipe.storage.redshift.management
  "Manage the database artifacts in redshift"
  (:require [clj-time.format        :as tf]
            [clojure.string         :as str]
            [clojure.java.jdbc      :as sql]
            [clojure.java.io        :as io]
            [clojure.tools.logging  :as log]))

(def ^{:private true} SHOW_RELATIONS_SQL
  ;; NOTE: USE of ILIKE for case-insensitive matching. (Postgres extension)
  "SELECT table_name, table_type
     FROM information_schema.tables
     WHERE table_schema = 'public' And
           table_name ilike ? AND
           table_type = ?
     ORDER BY table_name DESC")

(def date-formatter (:basic-date tf/formatters))

(defn- table-exists? [tname]
  (sql/with-query-results rows [SHOW_RELATIONS_SQL tname "BASE TABLE"]
    (doall rows)))


(defn- view-exists? [vname]
  (sql/with-query-results rows [SHOW_RELATIONS_SQL vname "VIEW"]
    (doall rows)))

(defn all-tables-seq [vname]
  (sql/with-query-results rows
    [SHOW_RELATIONS_SQL (str vname "%") "BASE TABLE"]
    (let [pattern (re-pattern (str "(?i)" vname "_\\d+"))
          data (mapv :table_name rows)]
      (filter (partial re-matches pattern) data)))) ;; need to filter more than the SQL for segment_ / segment_feed_

(defn- create-union-view [vname]
  (let [tables  (all-tables-seq vname)]
    (when (seq tables)
      (let [sql (apply str "CREATE OR REPLACE VIEW " vname " AS "
                       (interpose " UNION ALL "
                                  (mapv #(str "SELECT * FROM " %) tables)))]
        (log/debug sql)
        (sql/do-commands sql)))))

(defn- create-view [vname]
  (let [[latest & _]  (all-tables-seq vname)]
     (when latest
      (let [sql (str "CREATE OR REPLACE VIEW " vname " AS SELECT * FROM " latest)]
        (log/debug sql)
        (sql/do-commands sql)))))

(defn- create-table-if-not-exists [tname colspecs]
  (when-not (table-exists? tname)
    (log/debug (apply sql/create-table-ddl tname colspecs))
    (apply sql/create-table tname colspecs)))

(defn drop-table-if-exists [tname]
  (when (table-exists? tname)
    (sql/drop-table tname)))

(defn- truncate-table [tname]
  (sql/do-commands
   (str "TRUNCATE TABLE " tname)))

(defn truncate-table-if-exists [tname]
  (when (table-exists? tname)
    (truncate-table tname)))

(comment  ;;; DANGER WILL ROBINSON
          ;;; This will destroy your database!
  (defn drop-all-tables []
   (let [views (sql/with-query-results rows
                 [SHOW_RELATIONS_SQL "%" "VIEW"]
                 (mapv :table_name rows))
         tables (sql/with-query-results rows
                  [SHOW_RELATIONS_SQL "%" "BASE TABLE"]
                  (mapv :table_name rows))]
     (doseq [view views]
       (sql/do-commands (str "DROP VIEW " view)))
     (doseq [table tables]
       (sql/drop-table table)))))

(defn- grant-select-to [name & groups]
  (sql/do-commands
   (format "GRANT SELECT ON %s TO %s"
           name
           (str/join \, (map (partial str "GROUP ") groups)))))

(defn- format-table-name [name date]
  (format "%s_%s" name   (tf/unparse date-formatter date)))

(defn has-rows? [tname]
  (sql/with-query-results rows [(str "SELECT COUNT(*) from " tname " as count")]
    (pos? (:count (first rows)))))

(defn create-table
  "create the feed table"
  [db table-meta item]
  (let [{:keys [feed-name
                date]}    item
        table-name        (format-table-name feed-name date)
        colspecs          (get table-meta (.toLowerCase feed-name) [])]
    (log/info "Creating Table " table-name  "for feed " feed-name)
    (sql/with-connection db
      (truncate-table-if-exists table-name)
      (create-table-if-not-exists table-name colspecs)
      (grant-select-to table-name "datascientist" "operator"))
    table-name))

(defn update-view [db item]
  (let [{:keys [feed-name union-view?]} item]
   (sql/with-connection db
     (if union-view?
       (create-union-view feed-name)
       (create-view feed-name))
     (grant-select-to feed-name "datascientist" "operator"))))
