(ns kixi.comms.components.kafka
  (:require [cheshire.core :refer [parse-string]]
            [clojure.core.async :as async]
            [cognitect.transit :as transit]
            [com.stuartsierra.component :as component]
            [franzy.clients.consumer
             [callbacks :as callbacks]
             [client :as consumer]
             [defaults :as cd]
             [protocols :as cp]]
            [franzy.clients.producer
             [client :as producer]
             [defaults :as pd]
             [protocols :as pp]]
            [franzy.serialization
             [deserializers :as deserializers]
             [serializers :as serializers]]
            [kixi.comms :as comms]
            [kixi.comms.time :as t]
            [taoensso.timbre :as timbre :refer [error info]]
            [zookeeper :as zk])
  (:import [java.io ByteArrayInputStream ByteArrayOutputStream]))

;; https://github.com/pingles/clj-kafka/blob/321f2d6a90a2860f4440431aa835de86a72e126d/src/clj_kafka/zk.clj#L8
(defn brokers
  "Get brokers from zookeeper"
  [host port]
  (let [z (atom nil)]
    (try
      (reset! z (zk/connect (str host ":" port)))
      (if-let [broker-ids (zk/children @z "/brokers/ids")]
        (let [brokers (doall (map (comp #(parse-string % true)
                                        #(String. ^bytes %)
                                        :data
                                        #(zk/data @z (str "/brokers/ids/" %)))
                                  broker-ids))]
          (when (seq brokers)
            (mapv (fn [{:keys [host port]}] (str host ":" port)) brokers))))
      (finally
        (when @z
          (zk/close @z))))))

(defn clj->transit
  ([m]
   (clj->transit m :json))
  ([m t]
   (let [out (ByteArrayOutputStream. 4096)
         writer (transit/writer out t)]
     (transit/write writer m)
     (.toString out))))

(defn transit->clj
  ([s]
   (transit->clj s :json))
  ([s t]
   (let [in (ByteArrayInputStream. (.getBytes s))
         reader (transit/reader in t)]
     (transit/read reader))))

(defmulti format-message (fn [t _ _ _ _] t))

(defmethod format-message
  :command
  [_ command-key command-version payload _]
  {:kixi.comms.message/type       :command
   :kixi.comms.command/id         (str (java.util.UUID/randomUUID))
   :kixi.comms.command/key        command-key
   :kixi.comms.command/version    command-version
   :kixi.comms.command/created-at (t/timestamp)
   :kixi.comms.command/payload    payload})

(defmethod format-message
  :event
  [_ event-key event-version payload {:keys [origin]}]
  {:kixi.comms.message/type     :event
   :kixi.comms.event/id         (str (java.util.UUID/randomUUID))
   :kixi.comms.event/key        event-key
   :kixi.comms.event/version    event-version
   :kixi.comms.event/created-at (t/timestamp)
   :kixi.comms.event/payload    payload
   :kixi.comms.event/origin     origin})

(defn create-producer
  [in-chan topics origin broker-list]
  (async/go
    (try
      (let [key-serializer     (serializers/string-serializer)
            value-serializer   (serializers/string-serializer)
            pc                 {:bootstrap.servers broker-list}
            po                 (pd/make-default-producer-options)
            producer           (producer/make-producer
                                pc
                                key-serializer
                                value-serializer
                                po)]
        (loop []
          (let [msg (async/<! in-chan)]
            (if msg
              (let [[topic-key message-type message-version payload] msg
                    topic     (get topics topic-key)
                    formatted (apply format-message (conj msg {:origin origin}))
                    rm        (pp/send-sync! producer topic nil 
                                             (or (:kixi.comms.command/id formatted)
                                                 (:kixi.comms.event/id formatted))
                                             (clj->transit formatted) po)]
                (recur))
              (.close producer)))))
      (catch Exception e 
        (error e "Producer Exception")))))

(defn create-consumer
  [kill-chan out-chan group-id {:keys [commands events] :as topics} broker-list]
  (async/go
    (try
      (let [timeout            500
            key-deserializer   (deserializers/string-deserializer)
            value-deserializer (deserializers/string-deserializer)
            cc                 {:bootstrap.servers       broker-list
                                :group.id                group-id
                                :auto.offset.reset       :earliest
                                :enable.auto.commit      true
                                :auto.commit.interval.ms timeout}
            listener            (callbacks/consumer-rebalance-listener
                                 (fn [tps]
                                   (info "topic partitions assigned:" tps))
                                 (fn [tps]
                                   (info "topic partitions revoked:" tps)))
            co                 (cd/make-default-consumer-options
                                {:rebalance-listener-callback listener})
            consumer           (consumer/make-consumer
                                cc
                                key-deserializer
                                value-deserializer)
            running?            (atom true)]
        (cp/subscribe-to-partitions! consumer (vals topics))
        (async/go (async/<! kill-chan) (reset! running? false))
        (loop []
          (let [cr (into [] (cp/poll! consumer {:poll-timeout-ms timeout}))]
            (loop [cr' cr]
              (when-let [process (first cr')]
                (if @running?
                  (do
                    (async/put! out-chan ((comp transit->clj :value) process))
                    (recur (rest cr')))
                  (do
                    (cp/commit-offsets-sync! consumer {(select-keys process [:topic :partition])
                                                       {:offset (:offset process)
                                                        :metadata (str "Consumer stopping - "(java.util.Date.))}}))))))
          (if @running?
            (recur)
            (do
              (cp/clear-subscriptions! consumer)
              (.close consumer)))))
      (catch Exception e 
        (error e "Consumer exception")))))

(defn start-listening!
  [handlers {:keys [command event]} consumer-out-ch]
  (async/go-loop []
    (let [msg (async/<! consumer-out-ch)]
      (if msg
        (let [handlers' (get-in @handlers [(:kixi.comms.message/type msg)
                                           (or (:kixi.comms.command/key msg)
                                               (:kixi.comms.event/key msg))
                                           (or (:kixi.comms.command/version msg)
                                               (:kixi.comms.event/version msg))])]
          (run! (fn [f] (f msg)) handlers')
          (recur))))))

(defrecord Kafka [host port group-id topics origin]
  comms/Communications
  (send-event! [{:keys [producer-in-ch]} event version payload]
    (when producer-in-ch
      (async/put! producer-in-ch [:event event version payload])))
  (send-command! [{:keys [producer-in-ch]} command version payload]
    (when producer-in-ch
      (async/put! producer-in-ch [:command command version payload])))
  (attach-event-handler! [{:keys [handlers]} event version handler]
    (swap! handlers #(update-in % [:event event version] (fn [x] (conj x handler)))))
  (attach-command-handler! [{:keys [handlers]} command version handler]
    (swap! handlers #(update-in % [:command command version] (fn [x] (conj x handler)))))
  component/Lifecycle
  (start [component]
    (let [topics (or topics {:command "command" :event "event"})
          origin (or origin (.. java.net.InetAddress getLocalHost getHostName))
          broker-list        (brokers host port)
          producer-chan      (async/chan)
          consumer-kill-chan (async/chan)
          consumer-out-chan  (async/chan 1)
          handlers           (atom {:command {} :event {}})]
      (info "Starting Kafka Producer/Consumer")
      (create-producer producer-chan
                       topics
                       origin
                       broker-list)
      (start-listening! handlers topics consumer-out-chan)
      (create-consumer consumer-kill-chan
                       consumer-out-chan
                       group-id
                       topics
                       broker-list)
      (assoc component
             :handlers handlers
             :producer-in-ch producer-chan
             :consumer-kill-ch consumer-kill-chan
             :consumer-out-ch consumer-out-chan)))
  (stop [component]
    (let [{:keys [producer-in-ch
                  consumer-kill-ch
                  consumer-out-ch]} component]
      (info "Stopping Kafka Producer/Consumer")
      (async/close! producer-in-ch)
      (async/close! consumer-kill-ch)
      (async/close! consumer-out-ch)
      (dissoc component
              :handlers
              :producer-in-ch
              :consumer-kill-ch
              :consumer-out-ch))))
