(ns lotame.feed
  "Interact with lotame files"
  (:require [kixipipe.protocols         :as kixi]
            [kixipipe.transport.sftp    :as sftp]
            [kixipipe.ioplus            :as ioplus]
            [clj-time.format            :as tf]
            [clojure.set                :as set]
            [clojure.string             :as str]
            [clojure.tools.logging      :as log]
            [clojure.java.io            :as io]
            [schema.core                :as s]
            [com.stuartsierra.component :as component]
            [potemkin]))

(potemkin/import-macro kixipipe.transport.sftp/with-session)

;; FIXME - a better way for things to discover feeds.
(def FEED_NAME_OVERRIDES {"audience" "metadata"
                          "audience_membership" "audiencemembership"})
(def REV_FEED_NAME_OVERRIDES (set/map-invert FEED_NAME_OVERRIDES))

(def KNOWN_FEEDS #{"audience" "audience_membership" })

(def ^:private SRC_NAME "lotame")

(def ^:private lotame-date-formatter (tf/formatter "yyyyMMdd"))

(declare download-lotame-item!)
(defrecord LotameFeedItem [src-name feed-name dir filename checksum metadata submission-time]
  kixi/FeedItem
  (download! [this session]
    (download-lotame-item! session this))
  io/Coercions
  (as-file [item] (io/file (:dir item) (:filename item)))
  (as-url [item] (.toURI (io/as-file item))))

(declare lotame-feed-details)
(defrecord LotameSession [host user ssh-key dir download-dir]
  component/Lifecycle
  (start [this]
    (println "Starting Lotame Session.")
    this)
  (stop [this]
    (println "Stopping Lotame Session.")
    this)
  kixi/FeedSession
  (feed-details [this feed-name options]
    (condp #(%1 %2) feed-name
      KNOWN_FEEDS (lotame-feed-details this feed-name options)
      (throw (ex-info "Unknown Feed Name" nil)))))

(defn- files-with-done-marker [files]
  (let [md5-or-done (fn [[done? files] f]
                      (condp = (second (ioplus/base-and-ext (:filename f)))
                        "md5" [done? files]
                        "done" [true files]
                        [done? (conj files f)]))]
    (reduce md5-or-done [false nil] files)))

(defn- valid? [date]
  (try
    (tf/parse lotame-date-formatter date)
    (catch IllegalArgumentException e
      nil)))

(defn- md5-value [session dir filename]
  (first (str/split (sftp/sftp-get-as-string session (str dir "/" filename ".md5")) #"\s+")))

(def ^:private Config {:host String
                       :user String
                       :ssh-key (s/pred ioplus/exists-as-file?)
                       (s/optional-key :dir) (s/pred ioplus/exists-as-dir?)
                       s/Keyword String})

(defn mk-session
  "Creates a lotame session using the supplied config. The session should be passed to all
   lotame feed calls."
  [{:keys [host user ssh-key dir download-dir] :as config}]
  (s/validate Config config)
  (->LotameSession host user ssh-key dir download-dir))

(defn- lotame-feed-details
  "Retrieve details of feed, restricted by name and/or date."
  [session feed-name options]

  (let [{:keys [date]} options
        date-str             (when date (tf/unparse lotame-date-formatter date))
        [_ & files]          (sftp/sftp-file-seq session (or date-str ".")) ; first is always the root, so drop it
        dir-contents         (group-by :dir files)
        process-files-in-dir (fn [[date files]]
                               (when-let [date (valid? date)]
                                 (let [[done? files] (files-with-done-marker files)]
                                   (when done?
                                     (map (fn [{:keys [filename dir attrs]}]
                                            (if-let [[_ feed-name] (re-matches #"^(\w+)\..*$" filename)]
                                              (let [feed-name (get REV_FEED_NAME_OVERRIDES feed-name feed-name)]
                                                (map->LotameFeedItem
                                                 (merge options
                                                        {:src-name  SRC_NAME
                                                         :feed-name feed-name
                                                         :filename  filename
                                                         :date      date
                                                         :dir       dir
                                                         :checksum  (md5-value session dir filename)
                                                         :metadata  {:timestamp (:mtime attrs)}
                                                         :encoding  :gzip
                                                         :delimiter \tab}
                                                        (when (= feed-name "audience")
                                                          {:results-key :audience}))))))
                                          files)))))]
    (log/info "Retrieving details for feed " feed-name " with options " options)
    (doall (-> (mapcat process-files-in-dir dir-contents)
               (cond->> feed-name (filter #(-> % :filename (.startsWith (get FEED_NAME_OVERRIDES feed-name feed-name)))))
               (cond->> date-str (filter #(= date-str (:dir %))))))))

(defn- apply-name-overrides [item]
  (let [{:keys [feed-name]} item
        new-name (get REV_FEED_NAME_OVERRIDES feed-name feed-name)]
    (assoc item :feed-name new-name)))

(defn- download-lotame-item!
  "will download the feed item detailed by item and enrich the item with further details.
   Can be used to map over the results of feed-details.

   ```
   (doall (map (partial do-download! session) (feed-details ...)
   ```"
  [session item]
  (log/info "Downloading " item)
  (sftp/sftp-get-to-file session (apply-name-overrides item)))

(defn audience-feed-details
  "Get details of the audience feed. See `feed-details` for more info."
  [session date]
  (lotame-feed-details session "audience" date))

(defn audience-membership-feed-details
  "Get details of the audience-membership feed. See `feed-details` for more info."
  [session date]
  (lotame-feed-details session "audience_membership" date))

(comment

  (require '[controller.config  :as config])
  (require '[lotame.feed        :as lotame])
  (require '[anmedia.protocols  :refer [download! feed-details]])

  (-> (lotame/mk-session (config/load-config :prod))
      (sftp-start)
      (feed-details session "metadata" nil {})
      (sftp-stop))
  (lotame/with-session [session (lotame/mk-session (config/load-config :prod))]
    (feed-details session "metadata" nil {}))

)
