(ns google-dfp.dfp
  (:require [camel-snake-kebab.core :as csk]
            [clojure.data.csv :as csv]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [kixipipe.digest :as digest]
            [kixipipe.google.auth :as google-auth]
            [kixipipe.ioplus :as ioplus]
            [schema.core :as s])
  (:import [com.google.api.ads.dfp.axis.factory DfpServices]
           [com.google.api.ads.dfp.axis.utils.v201508 StatementBuilder ReportDownloader]
           [com.google.api.ads.dfp.axis.v201508 PublisherQueryLanguageServiceInterface Row SetValue DateTimeValue DateTime Date]
           [com.google.api.ads.dfp.lib.client DfpSession$Builder]
           [com.google.api.ads.dfp.lib.utils DateTimesHelper]))

(def ^:private DATETIMES_HELPER (DateTimesHelper. DateTime Date))

(def ^:private Config {:email-address String
                       :p12-file (s/pred ioplus/exists-as-file? "<credentials file>")
                       :network-code String
                       :bucket-name String
                       :download-dir (s/pred ioplus/exists-as-dir? "<a directory>")
                       (s/optional-key :page-size) int})

;; 'Table' definitions from:
;; https://developers.google.com/doubleclick-publishers/docs/reference/v201508/PublisherQueryLanguageService?hl=en
(def COLUMNS {"Geo_Target"                ["Id" "Name" "CanonicalParentId" "ParentIds" "CountryCode" "Type" "Targetable"]
              "Bandwidth_Group"           ["Id" "BandwidthName"]
              "Browser"                   ["Id" "BrowserName" "MajorVersion" "MinorVersion"]
              "Browser_Language"          ["Id" "BrowserLanguageName"]
              "Device_Capability"         ["Id" "DeviceCapabilityName"]
              "Device_Manufacturer"       ["Id" "MobileDeviceManufacturerName"]
              "Mobile_Carrier"            ["Id" "CountryCode" "MobileCarrierName"]
              "Mobile_Device"             ["Id" "MobileDeviceManufacturerId" "MobileDeviceName"]
              "Mobile_Device_Submodel"    ["Id" "MobileDeviceSubmodelName"]
              "Operating_System"          ["Id" "OperatingSystemName"]
              "Operating_System_Version"  ["Id" "OperatingSystemId" "MajorVersion" "MinorVersion" "MicroVersion"]
              "Third_Party_Company"       ["Id" "Name" "Type" "Status"]
              "Time_Zone"                 ["Id" "StandardGmtOffset" "SupportedInReports"]
              ;; For LineItem Targetting is ommitted. API docs have onerous warning about it being bleeding edge
              "LineItem"                  ["CostType" "CreationDateTime" "DeliveryRateType" "EndDateTime" "ExternalId" "Id" "IsMissingCreatives" "IsSetTopBoxEnabled" "LastModifiedDateTime" "LineItemType" "Name" "OrderId" "StartDateTime" "Status" "UnitsBought"]
              "Ad_Unit"                   ["AdUnitCode" "ExternalSetTopBoxChannelId" "Id" "LastModifiedDateTime" "Name" "ParentId" "PartnerId"]
              "User"                      ["Email" "ExternalId" "Id" "IsServiceAccount" "Name" "RoleId" "RoleName"]
              ;; FIXME - listed in the API docs, but 'No Such Table' Error'
              ;; "Exchange_Rate" ["CurrencyCode", "Direction", "ExchangeRateId","RefreshRate"]
              "Programmatic_Buyer"        ["AdxBuyerNetworkId" "BuyerId" "Name" "ParentId" "Type"]
              "Audience_Segment_Category" ["Id" "Name" "ParentId"]
              "Audience_Segment"          ["CategoryIds" "Id" "Name" "OwnerAccountId" "OwnerName" "SegmentType"]
              ;; FIXME - listed in the API docs, but 'No Such Table' Error'
              ;; "Proposal_Retraction_Reason" ["Id" "IsActive" "Name"]
              "Audience_Explorer"         ["Id" "ThirtyDayActiveSize" "ThirtyDayClicks" "ThirtyDayImpressions"]})

(defn- add-credential [{:keys [email-address p12-file] :as session} scopes]
  (assoc session :credential (google-auth/build-credential session scopes)))

(defn- add-dfp-service [{:keys [network-code credential] :as session}]
  (assoc session :dfp-service
         (-> (DfpSession$Builder. )
             ;; empty config prevents a series of exceptions being logged.
             ;; We prefer to read the config from our own config file.
             (.from (org.apache.commons.configuration.BaseConfiguration.))
             ;; update session with our values.
             (.withApplicationName "kixi data loader")
             (.withNetworkCode network-code)
             (.withOAuth2Credential credential)
             (.build))))

(defn- add-pql-service [{:keys [dfp-service] :as session}]
  (assoc session :pql-service (.get (DfpServices.)
                                    dfp-service
                                    PublisherQueryLanguageServiceInterface)))

(defrecord DfpMappingSession []
  component/Lifecycle
  (start [this]
    (println "Starting DfpMappingSession")
    (-> this
        (add-credential #{"https://www.googleapis.com/auth/dfp"})
        add-dfp-service
        add-pql-service))
  (stop [this]
    (println "Stopping GCSSession")
    this))

(defn mk-session [config]
  (s/validate Config config)
  (map->DfpMappingSession config))

(defn- build-statement [name page-size]
  (-> (StatementBuilder.)
      (.select (str/join "," (get COLUMNS name)))
      (.from (csk/->Camel_Snake_Case name))
      (.limit (int (or page-size StatementBuilder/SUGGESTED_PAGE_LIMIT)))
      (.offset (int 0))))

(defmulti value->string type)

(defmethod value->string :default [x]
  (.getValue x))

(defmethod value->string SetValue [x]
  (when-let [values (.getValues x)]
    (str/join "," (map #(.getValue %) values))))

(defmethod value->string DateTimeValue [x]
  (when-let [date-time (.getValue x)]
    (.toString DATETIMES_HELPER date-time)))

(defn- row->vec [x]
  (mapv value->string (.getValues x)))

(defn- paged-results [pql-service page-size stmt-builder]
  (log/debug "Paging...")
  (let [page (.select pql-service (.toStatement stmt-builder))
        rows (map row->vec (seq (.getRows page)))]
    (if (= (count rows) page-size)
      (lazy-cat rows
                (paged-results pql-service
                               page-size
                               (.increaseOffsetBy stmt-builder
                                                  page-size)))
      rows)))

(defn- output-file-for [download-dir mapping-name ]
  (io/file download-dir
           (format "%s-%d.csv" mapping-name (System/currentTimeMillis))))

(defn get-mapping-as-csv [{:keys [dfp] :as session} mapping-name]
  (log/info "Getting Mapping for " mapping-name)
  (let [{:keys [pql-service page-size download-dir]} dfp
        file                            (output-file-for download-dir mapping-name)]
    (with-open [out-md5 (digest/md5-output-stream  file)]
      (with-open [out (io/writer out-md5)]
        (csv/write-csv out
                       (concat [(get COLUMNS mapping-name)]
                               (paged-results pql-service page-size (build-statement mapping-name page-size)))
                       :quote? (constantly true)))
      {:feed-name mapping-name
       :dir       (.getParent file)
       :filename  (.getName file)
       :checksum (digest/md5-checksum-as-string out-md5)})))

(comment

    (def config {:email-address "128906095177-etsb0pp20a3i9i8hd9oh2nktlha3h1ec@developer.gserviceaccount.com"
                 :p12-file (io/as-file "/home/neale/Downloads/doubleclick-publishers-feed-da3f4a57bf24.p12")
                 :bucket-name "gdfp-5765"
                 :network-code "5765"
                 :download-dir (io/as-file "/tmp")
                 :page-size (int 2)})

    (def sess (component/start (mk-session config)))

    ;; sample data from all tables.
    (doseq [[name _] google-dfp.dfp/COLUMNS]
      (try
        (println name "\n======================")
        (clojure.pprint/pprint (take 5 (get-mapping-as-csv sess name)))
        (catch Throwable t
          (println "ERROR:" t))))

    )
