(ns google-dfp.feed
  "Read the Google Doublclick for Publishers feed"
  (:require [camel-snake-kebab.core :as csk]
            [cheshire.core :as json]
            [clj-time.core :as t]
            [clj-time.format :as tf]
            [clojure.java.io :as io]
            [clojure.set :as set]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [com.stuartsierra.component :as component]
            [google-dfp.dfp :as dfp]
            [kixipipe.data.scrub :as scrub]
            [kixipipe.digest :as digest]
            [kixipipe.google.auth :as google-auth]
            [kixipipe.ioplus :as ioplus]
            [kixipipe.misc :as misc]
            [kixipipe.protocols :as kixi]
            [kixipipe.storage.gcs :as gcs]
            [kixipipe.transport.sftp :as sftp]
            [schema.core :as s]
            [slingshot.slingshot :refer [try+ throw+]]))

(def ^:private SRC_NAME "google-dfp")

(def DATA_TRANSFER_FEEDS #{"NetworkActiveViews"
                           "NetworkActivities"
                           "NetworkClicks"
                           "NetworkImpressions"
                           "NetworkVideoConversions"

                           "NetworkBackfillActiveViews"
                           "NetworkBackfillActivities"
                           "NetworkBackfillClicks"
                           "NetworkBackfillImpressions"
                           "NetworkBackfillVideoConversions"})

(def MATCH_TABLE_FEEDS (into #{} (keys dfp/COLUMNS)))

(def KNOWN_FEEDS (set/union DATA_TRANSFER_FEEDS MATCH_TABLE_FEEDS))

(def FEED_NAME_OVERRIDES (into {} (map #(vector % (str/lower-case (csk/->kebab-case-string %))) KNOWN_FEEDS)))
(def REV_FEED_NAME_OVERRIDES (set/map-invert FEED_NAME_OVERRIDES))

; copied from clojure.core
(defmacro ^:private assert-args
  [& pairs]
  `(do (when-not ~(first pairs)
         (throw (IllegalArgumentException.
                  (str (first ~'&form) " requires " ~(second pairs) " in " ~'*ns* ":" (:line (meta ~'&form))))))
     ~(let [more (nnext pairs)]
        (when more
          (list* `assert-args more)))))

(defmacro with-session
  "authenticates if necessary and invokes body. Retries authentication
once if body fails with 401 Http status and then invokes body again."
   [binding & body]
   (assert-args
    (vector? binding) "a vector for binding"
    (= 2 (count binding)) "exactly 2 forms in binding vector")
  `(let [session# ~(binding 1)
         ~(binding 0) session#]
     (try
       (do ~@body)
       (catch com.google.api.client.googleapis.json.GoogleJsonResponseException e#
         (if (== (.getStatusCode e#) 401)
           ;; FIXME should be able to only auth to the correct session, but it's too difficult.
           (do (google-auth/authenticate (:dfp session#))
               (google-auth/authenticate (:storage session#))
               ~@body)
           (do
             ;; excessive debugging to work out what's going on.
             (log/error "JSONERROR: " e#)
             (log/error "JSONERROR.getDetails: " (.getDetails e#))
             (when (.getDetails e#)
               (log/error "JSONERROR.getDetails.getCode:" (.. e# getDetails getCode)))

             (log/error "JSONERROR: dfp: " (:dfp session#))
             (log/error "JSONERROR: dfp: " (:storage session#))
             (throw e#)))))))

(declare download-data-transfer-item!)
(defrecord
    ^{:doc "Item indicating an entry in Data Transfer Feed."}
    GoogleDfpDataTransferFeedItem [src-name feed-name date metadata submission-time]
    kixi/FeedItem
    (src-name [_] SRC_NAME)
    (download! [item session]
      (download-data-transfer-item! session item))
    io/Coercions
    (as-file [item] (io/file (:dir item) (:filename item)))
    (as-url [item] (.toURI (io/as-file item))))

(declare download-match-table-item!)
(defrecord
    ^{:doc "Item indicating data about a Match Table retrieved
            via the PQL Service"}
    GoogleDfpMatchTableFeedItem [feed-name
    start-element metadata] kixi/FeedItem
    (src-name [_] "google-dfp")
    (download! [item session]
      (download-match-table-item! session item)))

(declare data-transfer-feed-details)
(declare match-table-feed-details)

(defrecord GoogleDfpSession []
  component/Lifecycle
  (start [this]
    (println "Starting GoogleDfpSession")
    this)
  (stop [this]
    (println "Stopping GoogleDfpSession")
    this)
  kixi/FeedSession
  (feed-details [session feed-name options]
    (let [feed-name (get REV_FEED_NAME_OVERRIDES feed-name)]
      (condp #(%1 %2) feed-name
        DATA_TRANSFER_FEEDS (data-transfer-feed-details session feed-name options)
        MATCH_TABLE_FEEDS (match-table-feed-details session feed-name options)))))

(defn mk-session [config]
  ;;(s/validate Config config) - not needed, since child components validate
  (map->GoogleDfpSession config))

(defn- download-data-transfer-item!
  "will download the feed item detailed by item and enrich the item with further details.
Can be used to map over the results of feed-details.

```
(doall (map (partial do-download! session) (data-transfer-feed-details ...)
```" [{:keys [storage]} item]
  (log/info "Downloading " item)
  (merge item
         (gcs/get-object storage item)))

(def ^:private dfp-hour-formatter (tf/formatter "yyyyMMdd_HH"))

(defn- to-date-time-no-hours [dt]
  (when dt
    (apply t/date-time ((juxt t/year t/month t/day) dt))))

(defn- ->data-transfer-item [{:keys [feed-name name] :as item}]
  (when-let [[date] (next (re-matches (re-pattern (str feed-name
                                                       "_\\d+_(\\d+_\\d+)\\..*")) name))]
    (let  [dt (tf/parse dfp-hour-formatter date)]
      (map->GoogleDfpDataTransferFeedItem (-> item
                                              (assoc
                                                  :feed-name (get FEED_NAME_OVERRIDES feed-name)
                                                  :filename name
                                                  :date (to-date-time-no-hours dt))
                                              (assoc-in [:metadata :hour] (tf/unparse dfp-hour-formatter dt)))))))

(defn- data-transfer-feed-details
  "Using the given config retrieves a feed for given name and date and
returns a map with details of downloadable parts of that feed."
  [{:keys [storage]} feed-name options]
  (keep ->data-transfer-item
        (kixi/list-items storage SRC_NAME feed-name options)))

(defn- match-table-feed-details [session feed-name options]
  (when (MATCH_TABLE_FEEDS feed-name)
    [(map->GoogleDfpMatchTableFeedItem
      (merge options
             {:src-name SRC_NAME
              :feed-name (get FEED_NAME_OVERRIDES feed-name)
              :date (or (:date options) (to-date-time-no-hours (t/now)))}))]))

(defn- download-match-table-item! [session {:keys [feed-name] :as item}]
  (-> (dfp/get-mapping-as-csv session (get REV_FEED_NAME_OVERRIDES feed-name))
      (assoc :src-name SRC_NAME
             :feed-name feed-name)))
