(ns appnexus.http
  (:require [cheshire.core       :as json]
            [clj-http.client     :as http]
            [slingshot.slingshot :refer [try+ throw+]]
            [clojure.java.io     :as io]
            [clojure.string :as str]
            [clj-time.format :as tf]
            [kixipipe.ratelimit :as rl]))

(def ^:dynamic *app-nexus-http-debug* false)
(def ^:dynamic *session* {})

(def date-header-formatter (tf/formatter "EEE, dd MMM yyyy HH:mm:ss zzz"))

(defn- ->appnexus-url [session url]
  (str (:base-url session) url))

(defn wrap-appnexus-request
  [client]
  (fn [req]
    (let [token-store (:token-store *session*)
          token (when token-store @token-store)
          headers (:headers req)
          resp (client
                (-> req
                    (cond-> token (assoc-in [:headers "Authorization"] token))))]
      resp)))

(defn wrap-appnexus-response
  "Extracts the Appnexus response, checks it's status and strips the wrapper
   from the http response"
  [client]
  (fn [req]
    (let [response (client req)
          {:keys [body] :as response} response
          {an-response :response}     body
          {status :status}            an-response]
      (if an-response
        (condp = status
          "OK"     (assoc response :body an-response)
          (throw+ {:type (keyword "appnexus.http" (.toLowerCase (or status (:error_id an-response)))) :data an-response}))
        response))))

;; standard set + 401, we want our auth failed code to run in that case.
(def unexceptional-status?
  (conj http/unexceptional-status? 401))

;;; We slightly modify clj-http's wrap-exceptions to slurp the body in the
;;; event of an exception - Appnexus provides some useful info in the body.
;;; we also don't throw an exception for 401, rather let the UNAUTH propagate
;;; and be handled by wrap-appnexus-response.
(defn wrap-exceptions
  "Middleware that throws a slingshot exception if the response is not a
  regular response. If :throw-entire-message? is set to true, the entire
  response is used as the message, instead of just the status number."
  [client]
  (fn [req]
    (let [{:keys [status] :as resp} (client req)]
      (if (or (not (clojure.core/get req :throw-exceptions true))
              (unexceptional-status? status))
        resp
        (if (:throw-entire-message? req)
          (throw+ resp "clj-http: status %d %s" (:status %) (assoc resp :body (slurp (:body resp))))
          (throw+ resp "clj-http: status %s" (:status %)))))))

(def ^{:doc "standard clj-http middleware + wrap-appnexus" :private true }
  appnexus-middleware [http/wrap-request-timing
                       http/wrap-lower-case-headers
                       http/wrap-query-params
                       http/wrap-redirects
                       wrap-appnexus-request
                       http/wrap-url
                       http/wrap-decompression
                       http/wrap-input-coercion
                       http/wrap-output-coercion
                       wrap-exceptions
                       http/wrap-accept
                       http/wrap-accept-encoding
                       http/wrap-content-type
                       http/wrap-form-params
                       http/wrap-nested-params
                       http/wrap-method
                       wrap-appnexus-response])

(defn authenticate
  "Authenticates with appnexus an updates the session with the result"
  [session]
  (let [auth (select-keys session [:auth])] ; do this to avoid potential session data leak to appnexus.
    (http/with-middleware appnexus-middleware
      (let [{:keys [body] :as response}
            (http/post (->appnexus-url session "auth")
                       {:body (json/generate-string auth)
                        :as :auto         ; force coercion even
                                        ; after exception, since
                                        ; appnexus returns a JSON body
                                        ; with useful info.
                        :coerce :always})]
        (reset! (:token-store *session*) (:token body))
        response))))

(defmulti parse-rate-limit-header (fn [name _] name))
(defmethod parse-rate-limit-header :default [_ _ ] nil)


(defmethod parse-rate-limit-header "x-rate-limit-read" [name value]
  (vector :read-limit value))

(defmethod parse-rate-limit-header "date" [name value]
  (vector :date  (tf/parse date-header-formatter value)))

(defmethod parse-rate-limit-header "x-count-read" [name value]
  (vector :read-count
          (into {} (map (comp (fn [[k v]] (vector (keyword k) (Long/parseLong v))) next)
                        (re-seq #"(.+?):(\d+)(?:,|$)" value)))))

(defn extract-rate-limit-headers [{:keys [headers] :as resp}]
  (->> headers
       (keep (fn [[name value]]
               (parse-rate-limit-header name value)))
       (into {})))

(defn build-request [content-type query]
  (let [content-type (when content-type {:as content-type
                                         :coerce :always})
        query        (when-not (empty? query) {:query-params query})
        debug        (when *app-nexus-http-debug* {:debug true
                                                   :debug-body true
                                                   :throw-entire-message? true})]
    (merge content-type query debug)))

(defn service-get* [session name content-type query]
  (rl/take-token session)
  (http/with-middleware appnexus-middleware
    (let [resp (http/get (->appnexus-url session name)
                         (build-request content-type query))]
      (rl/update-statistics session (extract-rate-limit-headers resp))
      resp)))

(defn service-get [session name & [query]]
  (:body (service-get* session name :auto query)))

(defn service-get-string [session name & [query]]
  (:body (service-get* session name nil query)))

(defn service-get-stream [session name & [query]]
  (let [{:keys [body headers]} (service-get* session name :stream query)
        {:strs [content-length content-type]} headers]
    (condp = content-type
      "application/octet-stream" (if (pos? (Long/parseLong content-length))
                                   body
                                   (throw (ex-info "No data found - body is zero length" headers)))
      "application/json"         (let [json (json/parse-stream (io/reader body))
                                       {:keys [error service]} json]
                                   (throw (ex-info (str "Expected byte stream, got json:" json) json)))
      :default                   (throw (ex-info "Unexpected context type" content-type)))))
