(ns ring.mock.request
  "Functions to create mock request maps."
  (:require [clojure.string :as string]
            [ring.util.codec :as codec]
            [clojure.java.io :as io]
            [cheshire.core :as json]
            [pantomime.mime :refer [mime-type-of]]))

(defn- encode-params
  "Turn a map of parameters into a urlencoded string."
  [params]
  (if params
    (codec/form-encode params)))

(defn make-boundary []
  (str "---------------------------"
       (Long/toHexString (.getTime (java.util.Date.)))))

(defn make-segment [name data]
  (let [ostream (java.io.ByteArrayOutputStream.)
        head (if (instance? java.io.File data)
               (.getBytes (str "Content-Disposition: form-data; name=\"" name "\"; filename=\"" (.getName data) "\"\r\nContent-Type: " (mime-type-of data) "\r\n\r\n"))
               (.getBytes (str "Content-Disposition: form-data; name=\"" name "\"\r\n\r\n")))
        body (cond
               (string? data) (.getBytes data)
               (map? data) (.getBytes (json/generate-string data))
               (instance? java.io.File data) (let [file data
                                                   uri (.toURI file)
                                                   path (java.nio.file.Paths/get uri)]
                                               (java.nio.file.Files/readAllBytes path))
               :else (byte-array 0))
        tail (.getBytes "\r\n")]
    (.write ostream head)
    (.write ostream body)
    (.write ostream tail)
    (.toByteArray ostream)))

(defn encode-multipart [params boundary]
  (let [ostream (java.io.ByteArrayOutputStream.)
        head (.getBytes (str "--" boundary "\r\n"))
        tail (.getBytes (str "--" boundary "--\r\n"))
        segments (map #(apply make-segment %) params)
        body (interpose (.getBytes (str "--" boundary "\r\n")) segments)]
    (.write ostream head)
    (doseq [bytes body]
      (.write ostream bytes))
    (.write ostream tail)
    (.toByteArray ostream)))

(defn header
  "Add a HTTP header to the request map."
  [request header value]
  (let [header (string/lower-case (name header))]
    (assoc-in request [:headers header] (str value))))

(defn content-type
  "Set the content type of the request map."
  [request mime-type]
  (-> request
      (assoc :content-type mime-type)
      (header :content-type mime-type)))

(defn content-length
  "Set the content length of the request map."
  [request length]
  (-> request
      (assoc :content-length length)
      (header :content-length length)))

(defn- combined-query
  "Create a query string from a URI and a map of parameters."
  [request params]
  (let [query (:query-string request)]
    (if (or query params)
      (string/join "&"
                   (remove string/blank?
                           [query (encode-params params)])))))

(defn- merge-query
  "Merge the supplied parameters into the query string of the request."
  [request params]
  (assoc request :query-string (combined-query request params)))

(defn query-string
  "Set the query string of the request to a string or a map of parameters."
  [request params]
  (if (map? params)
    (assoc request :query-string (encode-params params))
    (assoc request :query-string params)))

(defmulti body
  "Set the body of the request. The supplied body value can be a string or
  a map of parameters to be url-encoded."
  {:arglists '([request body-value])}
  (fn [request x] (type x)))

(defmethod body String [request string]
  (body request (.getBytes string)))

(defmethod body (class (byte-array 0)) [request bytes]
  (-> request
      (content-length (count bytes))
      (assoc :body (java.io.ByteArrayInputStream. bytes))))

(defmethod body java.util.Map [request params]
  (if (some #(instance? java.io.File %) (vals params))
    (let [boundary (make-boundary)]
      (-> request
          (content-type (str "multipart/form-data; boundary=" boundary))
          (body (encode-multipart params boundary))))
    (-> request
        (content-type "application/x-www-form-urlencoded")
        (body (encode-params params)))))

(defmethod body nil [request params]
  request)

(defn request
  "Create a minimal valid request map from a HTTP method keyword, a string
  containing a URI, and an optional map of parameters that will be added to
  the query string of the URI. The URI can be relative or absolute. Relative
  URIs are assumed to go to http://localhost."
  ([method uri]
   (request method uri nil))
  ([method uri params]
   (let [uri    (java.net.URI. uri)
         host   (or (.getHost uri) "localhost")
         port   (if (not= (.getPort uri) -1) (.getPort uri))
         scheme (.getScheme uri)
         path   (.getRawPath uri)
         query  (.getRawQuery uri)
         request {:server-port    (or port 80)
                  :server-name    host
                  :remote-addr    "localhost"
                  :uri            (if (string/blank? path) "/" path)
                  :query-string   query
                  :scheme         (or (keyword scheme) :http)
                  :request-method method
                  :headers        {"host" (if port
                                            (str host ":" port)
                                            host)}}]
     (if (#{:get :head} method)
       (merge-query request params)
       (body request params)))))
