(ns kepler16.deploy
  (:require [clj-http.client :as client]
            [jsonista.core :as json]
            [clojure.java.io :as io]
            [clojure.xml :as xml]
            [clojure.zip :as zip]
            [clojure.data.xml :as data.xml]
            [clojure.data.zip.xml :as zip-xml]
            [mach.pack.alpha.skinny :as skinny]
            [kepler16.github :as github]
            [clojure.string :as str]))

(set! *warn-on-reflection* true)

(defn deploy-file [{:keys [group-id artifact repo-id repository-url jar-file pom]
                    :or {pom "pom.xml"
                         repo-id "github"
                         repository-url (str "https://maven.pkg.github.com/" group-id "/" artifact)}}]
  (str
   "mvn deploy:deploy-file"
   " -DrepositoryId=" repo-id
   " -Durl=" repository-url
   " -Dfile=" jar-file
   " -DpomFile=" pom))

(defn deploy-github-maven-package [{:keys [group-id artifact repo-id repository-url jar-file pom token version] :as package}]
  (if (github/package-exists? {:repo artifact
                               :org group-id
                               :version version
                               :token token})
    "echo \"Package already exists.\""
    (deploy-file package)))

(defn deploy-clojars-maven-package [package]
  (deploy-file (assoc package
                      :repository-url "https://clojars.org/repo")))

(defn root []
  (-> ^String (slurp "pom.xml")
      (.getBytes)
      (io/input-stream)
      xml/parse
      zip/xml-zip))

(defn pom->group-id [pom-zipper]
  (first
   (zip-xml/xml->
    pom-zipper
    :groupId
    zip-xml/text)))

(defn pom->artifact [pom-zipper]
  (first
   (zip-xml/xml->
    pom-zipper
    :name
    zip-xml/text)))

(defn pom->version [pom-zipper]
  (first
   (zip-xml/xml->
    pom-zipper
    :version
    zip-xml/text)))

(defn pom->set-vars [root {:keys [group-id artifact-id package-name version]}]
  (-> root
      (zip-xml/xml1-> :groupId)
      (zip/edit (fn [node] (assoc node :content [group-id])))
      (zip/up)

      (zip-xml/xml1-> :name)
      (zip/edit (fn [node] (assoc node :content [package-name])))
      (zip/up)

      (zip-xml/xml1-> :artifactId)
      (zip/edit (fn [node] (assoc node :content [(or artifact-id package-name)])))
      (zip/up)

      (zip-xml/xml1-> :version)
      (zip/edit (fn [node] (assoc node :content [version])))
      (zip/up)

      (zip/node)))

(defn pom! [{:as package}]
  (->> package
       (pom->set-vars (root))
       data.xml/indent-str
       println))

(defn deploy-clojars! [{}]
  (println
   (deploy-clojars-maven-package {:group-id (pom->group-id (root))
                                  :artifact (pom->artifact (root))
                                  :repo-id "clojars"
                                  :jar-file "target/lib.jar"
                                  :version (pom->version root)})))

(defn deploy! [{}]
  (println
   (deploy-github-maven-package {:token (System/getenv "GITHUB_TOKEN")
                                 :group-id (pom->group-id (root))
                                 :artifact (pom->artifact (root))
                                 :repo-id "github"
                                 :jar-file "target/lib.jar"
                                 :version (pom->version (root))})))

(defn build! [{}]
  (apply skinny/-main ["--no-libs" "--project-path" "target/lib.jar"]))
