(ns kepler16.deploy
  (:require
   [clojure.string :as str]
   [clojure.java.shell :as sh]
   [hf.depstar :as depstar]
   [kepler16.github :as github]
   [kepler16.pom :as pom]))

(set! *warn-on-reflection* true)

(defn deploy-file [{:keys [group-id artifact repo-id repository-url jar-file pom]
                    :or {pom "pom.xml"
                         repo-id "github"
                         repository-url (str "https://maven.pkg.github.com/" group-id "/" artifact)}}]
  (str
   "mvn deploy:deploy-file"
   " -DrepositoryId=" repo-id
   " -Durl=" repository-url
   " -Dfile=" jar-file
   " -DpomFile=" pom))

(defn deploy-github-maven-package [{:keys [group-id artifact repo-id repository-url jar-file pom token version] :as package}]
  (if (github/package-exists? {:repo artifact
                               :org group-id
                               :version version
                               :token token})
    "echo \"Package already exists.\""
    (deploy-file package)))

(defn deploy-clojars-maven-package [package]
  (deploy-file (assoc package
                      :repository-url "https://clojars.org/repo")))


(defn deploy-clojars!
  [{:keys [jar] :as options}]
  {:pre [(string? jar)]}
  (let [pom (pom/load-pom!)
        command
        (deploy-clojars-maven-package {:group-id (pom/pom->group-id pom)
                                       :artifact (pom/pom->artifact pom)
                                       :repo-id "clojars"
                                       :jar-file jar
                                       :version (pom/pom->version pom)})
        result
        (as-> command $
          (str/split $ #" ")
          (apply sh/sh $))]
    (println (:out result))
    (binding [*out* *err*]
      (println (:err result)))))

(defn deploy-github!
  [{:keys [jar] :as options}]
  {:pre [(string? jar)]}
  (let [pom (pom/load-pom!)
        command
        (deploy-github-maven-package {:token (System/getenv "GITHUB_TOKEN")
                                      :group-id (pom/pom->group-id pom)
                                      :artifact (pom/pom->artifact pom)
                                      :repo-id "github"
                                      :jar-file jar
                                      :version (pom/pom->version pom)})
        result
        (as-> command $
          (str/split $ #" ")
          (apply sh/sh $))]

    (println (:out result))
    (binding [*out* *err*]
      (println (:err result)))))


(defn build-lib-and-deploy-to-clojars! [options]
  (depstar/jar options)
  (deploy-clojars! options))
