(ns leiningen.kap
  (:require [clojure.string :as string]
            [clojure.java.io :as io]
            [leiningen.new.templates :refer [slurp-resource]]
            [leiningen.kap.eval :refer [eval-in-project]]
            [leiningen.kap.commands.generate :as generate-command]))

(defn- derive-command [command]
  (let [mapping {#{"generate" "g"} :generate
                 #{"help" "h"}     :help
                 #{"init" "i"}     :init
                 #{"migrate" "m"}  :migrate
                 #{"rollback" "r"} :rollback}
        command-lookup (fn [[k v]] (when (k command) v))]
    (or (some command-lookup mapping) :help)))

(defn- make-dirs [path]
  (if (io/make-parents (str path "/."))
    (println "Created directory:" (str "'" path "'"))))

(defmulti run
  (fn [project environment command args]
    (derive-command command)))

(defmethod run :migrate
  [project environment command args]
  (run project environment "init" args)
  (eval-in-project project
                   environment
                   'leiningen.kap.commands.migrate/run
                   args
                   'leiningen.kap.commands.migrate))

(defmethod run :rollback
  [project environment command args]
  (run project environment "init" args)
  (eval-in-project project environment
                   'leiningen.kap.commands.rollback/run
                   args
                   'leiningen.kap.commands.rollback))

(defmethod run :generate
  [project environment command args]
  (run project environment "init" args)
  (generate-command/run (some #{":example"} args) (remove #{":example"} args)))

(defmethod run :init
  [project environment command args]
  (doseq [path ["db/migrated" "db/migrations" "db/templates"]]
    (make-dirs path)))

(defmethod run :help
  [project environment command args]
  (println (slurp-resource "help.txt")))

(defn kap
  [project & [command & args]]
  (let [environment (get-in project [:kapooya :environment] "development")]
    (run project environment command args)))
