(ns leiningen.kap
  (:require [clj-time.core :as time]
            [clj-time.format :as time-format]
            [clojure.string :as string]
            [clojure.java.io :as io]
            [leiningen.new.templates :refer [name-to-path slurp-resource]]
            [leiningen.core.eval :as lein-eval]
            [leiningen.core.project :refer [merge-profiles]]
            [stencil.core :as mustache]))

(def ^:private migrations-directory "db/migrations/")

(defn- migrated-filepath [environment]
  (str "db/migrated/" environment ".txt"))

(defn- current-timestamp []
  (let [format (time-format/formatters :basic-date-time-no-ms)
        timestamp (time-format/unparse format (time/now))]
    (string/replace timestamp #"T|Z" "")))

(defn- slurp-template
  [name]
  (let [path (str "templates/" (string/lower-case name) ".clj")]
    (if (io/resource path)
      (slurp-resource path)
      (let [file (io/file path)]
        (if (.exists file)
          (slurp file))))))

(defn- render-template
  [name params]
  (if-let [template (slurp-template name)]
    (mustache/render-string template params)))

(defn- filepath-to-name-space-string
  [filepath]
  (-> filepath (string/replace ".clj" "") (string/replace "/" ".") (string/replace "_" "-")))

(defn- migration-file-reader [environment]
  (let [path (migrated-filepath environment)
        file (io/file path)]
    (if-not (.exists file)
      (.createNewFile file))
    (io/reader path)))

(defn- previous-migrations [environment]
  (with-open [r (migration-file-reader environment)]
    (apply sorted-set (remove clojure.string/blank? (line-seq r)))))

(defn- name-to-path-section
  [name]
  (-> (string/replace name #"[^a-zA-Z]" " ")
      string/trim
      (string/replace #"\s+" "_")
      (string/replace #"_+" "_")))

(defn- template-name-from
  [args]
  (->> (map name-to-path-section args)
       (cons (current-timestamp))
       (string/join "_" )
       name-to-path))

(defn- find-by-set-key
  [mapping key]
  (some (fn [[k v]] (when (k key) v)) mapping))

(defn- migration-files []
  (->> (io/file migrations-directory) .listFiles (map str) sort (filter #(.endsWith % ".clj"))))

(defn- load-file-and-apply-fn
  [path f & args]
  (load-file path)
  (let [fn-name  (str (filepath-to-name-space-string path) "/" f)
        arg-str  (string/join " " args)
        eval-str (str "(" fn-name " " arg-str ")")]
    (load-string eval-str)))

(defn- generate
  [example? [template-name & template-args :as args]]
  (let [basename  (template-name-from args)
        filepath  (str migrations-directory basename ".clj")
        namespace (filepath-to-name-space-string filepath)
        params    (into {:namespace namespace :example example?} (map-indexed #(vector (str "arg" %1) %2) template-args))
        content   (render-template template-name params)]
    (if content
      (do
        (io/copy content (io/file filepath))
        (println "New migration file:" filepath))
      (println (format "Unable to locate template: '%s'" template-name)))))

(defn- make-dirs [path]
  (if (io/make-parents (str path "/."))
    (println "Created directory:" (str "'" path "'"))))

(defn- has-dependency? [project name]
  (->> project
       :dependencies
       (map first)
       (some #{name})
       boolean))

(defn- add-dependency [project dependency]
  (merge-profiles project [{:dependencies [dependency]}]))

(defn- add-missing-dependencies [project dependencies]
  (let [missing-dependencies (remove (comp (partial has-dependency? project) first)
                                     dependencies)]
    (reduce add-dependency project missing-dependencies)))

(defn- eval-in-project [project environment function args]
  (let [dependencies [['org.clojure/java.jdbc "0.3.0-alpha5"]
                      ['org.postgresql/postgresql "9.2-1003-jdbc4"]
                      ['leiningen "2.3.3"]
                      ['kapooya/kap "0.1.1"]]
        project (add-missing-dependencies project dependencies)]
    (lein-eval/eval-in-project project
                               `(~function '~project '~environment '~args)
                               '(require 'leiningen.kap))))

(defn migrate [project environment args]
  (let [migrations (migration-files)
        db-spec (-> project :kapooya :db-spec)]
    (doseq [filepath (remove (previous-migrations environment) migrations)]
      (load-file-and-apply-fn filepath 'up db-spec)
      (spit (migrated-filepath environment) (str filepath "\n") :append true)
      (println " ⬆ Applied migration:" filepath))))

(defn rollback [project environment args]
  (let [migrations (migration-files)
        db-spec (-> project :kapooya :db-spec)
        migrations (previous-migrations environment)
        last-migration (last migrations)]
    (if last-migration
      (do
        (load-file-and-apply-fn last-migration 'down db-spec)
        (spit (migrated-filepath environment) (str (string/join "\n" (disj migrations last-migration)) "\n"))
        (println " ⬇ Applied rollback:" last-migration)))))

(defmulti run
  (fn [project environment command args]
    (let [mapping {#{"generate" "g"} :generate
                   #{"help" "h"}     :help
                   #{"init"}         :init
                   #{"migrate" "m"}  :migrate
                   #{"rollback" "r"} :rollback}]
      (or (find-by-set-key mapping command) :help))))

(defmethod run :migrate
  [project environment command args]
  (run project environment "init" args)
  (eval-in-project project environment 'leiningen.kap/migrate args))

(defmethod run :rollback
  [project environment command args]
  (run project environment "init" args)
  (eval-in-project project environment 'leiningen.kap/rollback args))

(defmethod run :generate
  [project environment command args]
  (run project environment "init" args)
  (generate (some #{":example"} args) (remove #{":example"} args)))

(defmethod run :init
  [project environment command args]
  (doseq [path ["db/migrated" "db/migrations" "templates"]]
    (make-dirs path)))

(defmethod run :help
  [project environment command args]
  (println
  "
    Kapooya: Database migrations for Clojure projects

    Usage:

    lein kap help
      Print a short help message (default)

    lein kap init
      Creates directories that are required by Kapooya (Nondestructive).

    lein kap migrate
      Run the `up` method for all outstanding migrations sorted by filename (which starts with a timestamp).

    lein kap rollback
      Run the `down` method for the most recent migration.

    lein kap generate custom
      Creates a new migration shell ready to be completed with some custom code.

    lein kap generate create-table table-name
      Creates a new migration to create a table.

    lein kap generate drop-table table-name
      Creates a new migration to drop a table.

    lein kap generate add-column table-name column-name [column-type]
      Creates a new migration to add a column to a table. Column type defaults to `character varying(255)`.

    lein kap generate drop-column table-name column-name [column-type]
      Creates a new migration to drop a column from a table.

    lein kap generate create-index table-name column-name
      Creates a new migration to create an index.

    lein kap generate drop-index table-name column-name
      Creates a new migration to drop an index.
  "))

(defn kap
  [project & [command & args]]
  (let [environment (get-in project [:kapooya :environment] "development")]
    (run project environment command args)))
