package crux.api;

import java.io.Closeable;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Represents the database as of a specific valid and
 * transaction time.
 */
public interface ICruxDatasource {
    /**
     * Returns the document map for an entity.
     *
     * @param eid an object that can be coerced into an entity id.
     * @return    the entity document map.
     */
    public Map entity(Object eid);

    /**
     * Returns the entity tx for an entity.
     *
     * @param eid an object that can be coerced into an entity id.
     * @return    the entity transaction details.
     */
    public Map entityTx(Object eid);

    /**
     * Returns a new snapshot allowing for lazy query results in a
     * try-with-resources block using {@link #q(Closeable snapshot,
     * Object query)}. Can also be used for {@link
     * #historyAscending(Closeable snapshot, Object eid)} and {@link
     * #historyDescending(Closeable snapshot, Object eid)}
     *
     * @return an implementation specific snapshot
     */
    public Closeable newSnapshot();

    /**
     * Queries the db.
     *
     * @param query the query in map, vector or string form.
     * @return      a set or vector of result tuples.
     */
    public Collection<List> q(Object query);

    /**
     * Queries the db lazily.
     *
     * @param snapshot a snapshot from {@link #newSnapshot()}.
     * @param query    the query in map, vector or string form.
     * @return         a lazy sequence of result tuples.
     */
    public Iterable<List> q(Closeable snapshot, Object query);

    /**
     * Retrieves entity history lazily in chronological order from and
     * including the valid time of the db while respecting
     * transaction time. Includes the documents.
     *
     * @param snapshot a snapshot from {@link #newSnapshot()}.
     * @param eid      an object that can be coerced into an entity id.
     * @return         a lazy sequence of history.
     */
    public Iterable<Map> historyAscending(Closeable snapshot, Object eid);

    /**
     * Retrieves entity history lazily in reverse chronological order
     * from and including the valid time of the db while respecting
     * transaction time. Includes the documents.
     *
     * @param snapshot a snapshot from {@link #newSnapshot()}.
     * @param eid      an object that can be coerced into an entity id.
     * @return         a lazy sequence of history.
     */
    public Iterable<Map> historyDescending(Closeable snapshot, Object eid);

    /**
     * The valid time of this db.
     *
     * @return the valid time of this db.
     */
    public Date validTime();

    /**
     * The transaction time of this db.
     *
     * @return the transaction time of this db.
     */
    public Date transactionTime();
}
