(ns utilities.testing
  (:require
    [clojure.tools.namespace.find :as find]
    [clojure.test :refer [run-tests]]
    [utilities.db :as db]
    [utilities.settings :as settings]
    [utilities.migrations :as m]
    [clojure.java.jdbc :as jdbc]
    [clojure.java.io :as io]
    [clojure.string :as s]
    [utilities.exceptions :as ex]
    [peridot.core :as peridot]))


(defn redirects?
  "Returns true if the response redirects to given location"
  [res loc]
  (let [uri   (get (:headers res) "Location")
        r-loc (str "http://localhost" loc)]
    (or (= loc uri) (= r-loc uri))))


(defn request
  "Mock request in a session"
  ([session method uri] (request session method uri nil))

  ([session method uri params]
   (peridot/request session
                    uri
                    :request-method method
                    :params params)))

(defn new-session
  "Creates a new peridot session"
  [handler user]
  (let [session   (peridot/session handler)]
    (if user
      (let [resp (request session :post "/login/" user)]
        (assert (redirects? (:response resp) "/")
                (str "User Login failed in new session\n"
                     (:response resp)))
        resp)
      (request session :get "/"))))


(defn- run-on-all-tables
  "Runs a command on all tables in database"
  [command]
  (let [tables (jdbc/query (settings/db-spec)
                           ["SELECT table_name
                             FROM information_schema.tables
                             WHERE table_schema = ?" settings/mysql-db])
        user-commands (map (fn [{:keys [table_name]}]
                             (str command " " table_name))
                           tables)
        all-commands (concat ["SET FOREIGN_KEY_CHECKS = 0"]
                             user-commands
                             ["SET FOREIGN_KEY_CHECKS=1"])]
    (println "ALL COMMANDS" all-commands)
    (jdbc/db-do-commands (settings/db-spec) all-commands)))


(defn rollback
  "Rolls-back everything by running tests inside a transaction"
  [test]
  (jdbc/with-db-transaction [t-conn (settings/db-spec)]
                            (jdbc/db-set-rollback-only! t-conn)
                            (with-redefs [db/connection t-conn]
                              (test))))


(defn reset-db
  "Drops all tables in database"
  []
  (run-on-all-tables "DROP TABLE IF EXISTS"))


(defn truncate
  "Truncate all tables"
  ([]
   (truncate (fn [] (println "Truncated all data"))))
  ([test]
   (run-on-all-tables "TRUNCATE TABLE")
   (test)))


(defn run
  "Parse argument for ns name
  Rerun all migrations if :reset"
  ([] (run ""))
  ([pattern]
   (let [reset?   (= pattern ":reset")]
     (with-redefs [settings/in-testing?     true
                   settings/in-production?  true
                   settings/mysql-db        (str "test_" settings/mysql-db)]
       (db/enable-pooling)
       ; run migrations
       (when reset?
         (reset-db)
         (let [pending-m  (m/pending-count)]
           (when (> pending-m 0)
             (println "RUNNING " pending-m " pending migrations")
             (m/migrate))))

       ; load tests in given pattern
       (let [ns (find/find-namespaces-in-dir (io/file "test"))
             ns (filter #(s/includes? (str %) pattern) ns)
             ns (map (fn [ns] (require ns) (find-ns ns)) ns)]
         (apply run-tests ns))))))
