(ns utilities.settings
  (:require
    [clojure.string :as s]
    [taoensso.timbre :as timbre]))


(def in-production? (System/getenv "LEIN_NO_DEV"))

; Bound to true during testing
(def in-testing? false)

; Project Settings
(def salt (or (System/getenv "JURASSIC_SALT")
              "SecRet!#12343@Salt"))

; Database Settings
(def mysql-db (System/getenv "JURASSIC_MYSQL_DATABASE"))
(def mysql-host-port (System/getenv "JURASSIC_MYSQL_HOST"))
(def mysql-username (System/getenv "JURASSIC_MYSQL_USER"))
(def mysql-password (System/getenv "JURASSIC_MYSQL_PASSWORD"))

; Email Settings
(def email-default-from (System/getenv "JURASSIC_EMAIL_DEFAULT_FROM"))
(def email-host (System/getenv "JURASSIC_EMAIL_HOST"))
(def email-port (when-let [port (System/getenv "JURASSIC_EMAIL_PORT")]
                  (Integer. port)))
(def email-admins (when-let [admins (System/getenv "JURASSIC_EMAIL_ADMINS")]
                    (s/split admins #",")))
(def email-tls true)
(def email-spec {:host email-host :port email-port})


; Helper Functions
(defn db-spec
  "Returns db-spec required for database operations"
  []
  {:classname   "com.mysql.cj.jdbc.Driver"
   :subprotocol "mysql"
   :subname     (str "//" mysql-host-port "/" mysql-db "?useSSL=false")
   :user        mysql-username
   :password    mysql-password})


(defn memoize-in-prod
  "Memoizes a function in production"
  [f]
  (if in-production?
    (memoize f)
    f))

; Set initial defaults
(defn set-defaults
  "Set initial defaults"
  []
  ; Register clj-time jdbc protocol extension
  (require '[clj-time.jdbc])

  ; Configure timbre ns referrals
  (timbre/refer-timbre)

  (when-not (System/getenv "TIMBRE_LEVEL")
    (if in-production?
      (timbre/set-level! :warn)
      (timbre/set-level! :info))))

(set-defaults)
