(ns utilities.forms.core
  (:require [hiccup.core :refer [html h]]
            [ring.util.anti-forgery :refer [anti-forgery-field]]
            [clojure.string :as s]
            [utilities.forms.components :as field]
            [utilities.forms.validations :as v]))


(defn csrf-token
  "Alias for anti-forgery-field"
  []
  (anti-forgery-field))

; Field types

(def input-field ::field/input)
(def custom-field ::field/custom-field)
(def textarea-field ::field/textarea)
(def checkbox-field ::field/checkbox)
(def hidden-field ::field/hidden)
(def select-field ::field/select)


(defn form-field
  "Generates a field map with proper defaults"
  ([field-name] (form-field field-name {}))

  ([field-name {:keys [field-type label attrs help-msg optional?
                       renderer options]}]

   (when (and (= field-type custom-field)
              (not renderer))
     (throw (Exception.
              (str "Renderer not provided for custom field " field-name))))

   (when (and renderer (not= field-type custom-field))
     (throw (Exception. "Renderer is supported only for custom field")))

   (when (and (= field-type select-field)
              (not (map? options)))
     (throw (Exception.
              (str "No options provided for select field " field-name))))

   {:field-type (or field-type input-field)
    :name       (keyword field-name)
    :label      (or label (s/capitalize (name field-name)))
    :attrs      attrs
    :help-msg   help-msg
    :optional?  optional?
    :renderer   renderer
    :options    options}))


(defn render-form
  "renders form with errors and data"
  [fields data validation & options]
  (let [options     (set options)
        errors      (:errors validation)
        fields-html (for [field fields]
                      (let [key (:name field)]
                        (field/render field (key data) (key errors))))
        form-msg (when-let [msg (::form errors)]
                   (html [:div.alert.error (h msg)]))]
    (str form-msg
         (when-not (::without-csrf-token options)
           (str (anti-forgery-field) "\n"))
         (s/join "\n" fields-html))))


(defn get-fields-data
  "Returns map of field names and corresponding data from data map"
  [fields data]
  (into {} (for [field fields]
             (let [key (:name field)]
               [key (get data key
                         (get data (name key)))]))))


(defn get-post-data
  "Parses post data from request and returns matching map based on fields
  If initial data is provided, then post data is merged into it."
  ([fields request]
   (let [post-data (:form-params request)]
     (if-not (empty? post-data)
       (get-fields-data fields post-data)))))


; Form validation is responsible for checking
; - optional fields
; - error messages
; - mapping messages to fields


(defn get-error
  "Returns map of key and error if validation result is not clean"
  [key validation]
  (when-not (:is-clean? validation)
    {key (:error validation)}))


(defn validate-required
  "Validates non optional fields"
  [fields data]
  (let [fields  (remove :optional? fields)
        errors  (for [field fields]
                  (let [name   (if (keyword? field)
                                 field
                                 (:name field))
                        value  (get data name)]
                    (get-error name (v/validate-required value))))
        errors  (apply merge errors)]
    (if errors
      {:errors errors}
      {:is-clean? true})))


(defn call
  "Calls given validation if it is a function"
  [validation value]
  (let [result (if (fn? validation)
                 (validation value)
                 validation)]
    (if (and (not (contains? result :error))
             (not (contains? result :is-clean?)))
      (throw (Exception. (str "Bad validation result for " validation))))
    result))


(defn get-field-with-name
  "Returns field with given name.
  If the given name is already a field, then it returns itself."
  [fields name]
  (if (:name name)
    name
    (first (filter #(= (:name %) name)
                   fields))))


(defn run-form-validations
  "Runs form-validations in order unless any of them fails.
  Returns {:is-clean? true} if all validations pass."
  [data & form-validations]
  (loop [validations form-validations]
    (let [result (call (first validations) data)]
      (if-not (:is-clean? result)
        {:errors {::form (:error result)}}
        (if-let [validations (seq (rest validations))]
          (recur validations)
          {:is-clean? true})))))

(comment
  (run-form-validations
    {:email "foo@bar.com"}
    (v/validate-email "foo@bar.com")))


(defn run-field-validations
  "Runs field validations. If the validation is a function, then the function is called with field value
  Optional fields are checked only if they have a value."
  [data fields field-validations]
  (let [field-validations (if (map? field-validations)
                            (apply concat field-validations)
                            field-validations)
        errors  (for [[name validation] (partition 2 field-validations)]
                  (let [field  (get-field-with-name fields name)
                        name   (:name field)
                        value  (get data name)
                        result (if (and (:optional? field) (empty? value))
                                 {:is-clean? true}
                                 (call validation value))]
                    (get-error name result)))
        errors  (apply merge errors)]
    (if errors
      {:errors errors}
      {:is-clean? true})))


(comment
  (run-field-validations
    {:email "foo"}
    [(form-field :email)]
    {:email v/validate-email}))


(defmacro validate
  "Returns a map of :errors and :is-clean?"
  [data fields field-validations & form-validations]
  `(if (nil? ~data)
     nil
     (let [data#    ~data
           fields#   ~fields
           errors#  (merge (:errors (validate-required fields# data#))
                           (:errors (run-field-validations data#
                                                           fields#
                                                           ~field-validations)))]
       (if errors#
         {:errors errors#}
         (run-form-validations data# ~@form-validations)))))
