(ns utilities.db
  (:require [clojure.java.jdbc :as jdbc]
            [utilities.settings :as settings]
            [clojure.string :as s]
            [utilities.templates :as t]
            [utilities.exceptions :as ex])
  (:import (com.zaxxer.hikari HikariDataSource)))


; Function to BUILD a SELECT query
(defn seq-of-seq?
  [x]
  (and (sequential? x)
       (sequential? (first x))))

(defn collect-params
  ([clause]
   (cond
     (not clause) ["" nil]
     (string? clause) [clause nil]
     (sequential? clause) (let [[sql & params] clause]
                            [sql params])
     :else (throw (Exception. (str "Unable to parse " clause)))))
  ([clauses sep]
   {:pre (seq-of-seq? clauses)}
   (reduce (fn [[sql params] [s p]]
             [(str sql sep s) (concat params p)])
           (map collect-params clauses))))

(comment
  (collect-params [["foo = ? and bar = ?" 2 3]]
                  ", "))

(defn to-seq-of-seq
  "Converts input to seq of seq"
  [x]
  (cond
    (not x) nil
    (string? x) [[x]]
    (seq-of-seq? x) x
    (sequential? x) [x]
    :else (throw (Exception. (str "Unable to parse " x)))))


(defn- integer
  "Converts given value to number else nil"
  [x]
  (when x
    (Integer. x)))


(defn query
  "Create a SQL query map"
  ([from {:keys [select where group-by having order-by limit offset]}]
   (let []
     {:select    (seq (concat (:select from) (to-seq-of-seq select)))
      :from      (get from :from from)
      :where     (seq (concat (:where from) (to-seq-of-seq where)))
      :group-by  (or group-by (:group-by from))
      :having    (or having (:having from))
      :order-by  (or order-by (:order-by from))
      :limit     (or (integer limit) (:limit from))
      :offset    (or (integer offset) (:from offset))})))


(defn sql
  "Converts query map to SQL string"
  [q]
  (let [select (let [clauses (or (:select q) [["*"]])
                         [sql params] (collect-params clauses ", ")]
                     [(str "SELECT " sql) params])
        from (let [[sql params] (collect-params (:from q))]
                   [(str "FROM " sql) params])
        where (when-let [clauses (:where q)]
                    (let [[sql params] (collect-params clauses " AND ")]
                      [(str "WHERE " sql) params]))
        group-by (when-let [clause (:group-by q)]
                       (let [[sql params] (collect-params clause)]
                         [(str "GROUP BY " sql) params]))
        having (when-let [clause (:having q)]
                     (let [[sql params] (collect-params clause)]
                       [(str "HAVING " sql) params]))
        order-by (when-let [clause (:order-by q)]
                       (let [[sql params] (collect-params clause)]
                         [(str "ORDER BY " sql) params]))
        limit (when-let [n (:limit q)]
                    ["LIMIT ?" (list n)])
        offset (when-let [n (:offset q)]
                     ["OFFSET ?" (list n)])
        parts (remove nil? [select from where group-by
                            having order-by limit offset])
        [sql params] (reduce (fn [[res-sql res-p] [part-sql part-p]]
                               [(str res-sql "\n" part-sql)
                                (concat res-p part-p)])
                       parts)
        ; replace nil params with ""
        params  (for [p params]
                  (if-not p "" p))]
    (cons sql params)))

(comment
  (sql (query "foo" {:where ["id = ?" 1]})))


; Connection Pooling
; We are using dynamic connection to enable transactions when needed

(def ^:dynamic connection (settings/db-spec))

(defn get-pooled-connection
  "Creates Database connection pool to be used in queries"
  []
  (let [{:keys [subname user password]} (settings/db-spec)
        pool (doto (HikariDataSource.)
               (.setJdbcUrl (str "jdbc:mysql:" subname))
               (.setUsername user)
               (.setPassword password))]
    {:datasource pool}))

(defn enable-pooling
  "Sets connection to use pooled connection"
  []
  (def ^:dynamic connection (get-pooled-connection)))


; Functions to EXECUTE the query
; Documentation for jdbc api: https://clojure.github.io/java.jdbc/#clojure.java.jdbc/execute!

(defn get!
  "Returns first result for the given query.
  Raises error if none or more than one result exist."
  [q]
  (let [result    (jdbc/query connection (sql q))
        n         (count result)]
    (cond
      (= n 1) (first result)
      (> n 1) (ex/raise ::db-error "Found Multiple rows")
      (= n 0) (ex/raise ::db-error "No results found"))))


(defn get-or-404
  "Raises page-not-found error if get! fails"
  [q]
  (try
    (get! q)
    (catch Exception e
      (if (ex/cause? e ::db-error)
        (ex/raise ::ex/page-not-found "Resource not found")
        (throw e)))))


(defn first!
  "Limits and returns first record for given query
  Can return nil"
  [q]
  (let [q     (assoc q :limit 1)]
    (first (jdbc/query connection (sql q)))))


(defn count!
  "Runs MySQL count query for given query"
  [q]
  (let [q     (assoc q :select ["COUNT(*) as __count"])
        res   (jdbc/query connection (sql q))]
    (:__count (first res))))


(defn all!
  "Returns all results for the query"
  [q]
  (jdbc/query connection (sql q)))


(defn create
  "Create new table row"
  [table row]
  (try
    (let [res (jdbc/insert! connection table row)]
      (get! (query table
                   {:where ["id = ?" (:generated_key (first res))]})))
    (catch java.sql.SQLIntegrityConstraintViolationException e
      (if (s/includes? (ex/message e) "Duplicate entry")
        (ex/raise ::db-duplicate-error "Record already exists")
        (throw e)))))


(defn update!
  "Update table row where the clause matches"
  ([table set-map where]
   (jdbc/update! connection table set-map where)))


(defn insert-or-update
  "Performs insert-on-duplicate-update query"
  ([table data update-columns]
   (let [column-names  (keys (first data))
         values        (map vals data)]
     (insert-or-update table column-names values update-columns)))

  ([table column-names values update-columns]
   (let [update-columns     (map name update-columns)
         column-names       (map name column-names)
         update-columns-ref (s/join ", " (for [col update-columns]
                                           (str col " = VALUES(" col ")")))
         q-str (str "INSERT INTO "
                    table " ("
                    (s/join ", " column-names)
                    ") VALUES ("
                    (s/join ", " (repeat (count column-names) "?"))
                    ") ON DUPLICATE KEY UPDATE "
                    update-columns-ref)
         query (cons q-str values)]
     (try
       (jdbc/execute! connection query {:multi? true})
       (catch Exception e
         (println "QUERY TRIED: " query)
         (throw e))))))


(defn delete!
  "Delete table row where the clause matches"
  [table where] (jdbc/delete! connection table where))


(defmacro using-transaction
  "Runs given commands in a new transaction connection"
  [& body]
  `(jdbc/with-db-transaction [t-conn# (settings/db-spec)]
     (binding [connection t-conn#]
       ~@body)))
