(ns utilities.forms.core-test
  (:require [clojure.test :refer :all]
            [utilities.forms.components :as c]
            [utilities.exceptions :as ex]
            [utilities.forms.checks :as v]
            [utilities.forms.core :as f]
            [ring.mock.request :as mock]))


(defn- name-is-foo?
  [data]
  (when-not (= "Foo" (:name data))
    (ex/raise :validation-error "Please check the name"))
  true)

(def name-field (f/form-field "name"))
(def password-field (f/form-field "password"))

(def test-fields [name-field password-field])

(defn get-test-fields-errors
  [data]
  (f/checks
    (f/check-non-blank test-fields data)
    ::f/form  (name-is-foo? data)))


(deftest test-get-post-data
  (testing "with mock form"
    (let [request   {:form-params {"name" "foo", "bad" "abc"}}
          fields    [name-field password-field]
          data      (f/get-post-data fields request)]
      (is (= data {:name "foo" :password nil})))))

(deftest test-fill-form

  (testing "form with no checks"
    (let [fields  [name-field]
          data    {:name "foo"}
          errors  (f/checks (f/check-non-blank fields data))]
      (is (= errors :no-errors)))

  (testing "fill-form"
    (let [person  {:id 1, :name "Harry", :password "bar"}
          data    (f/get-fields-data test-fields person)]
      (is (= data {:name "Harry" :password "bar"}))))

  (testing "incremental fill"
    (let [default   {:name "Harry", :password "bar"}
          i         {:password "baz"}
          data      (f/get-fields-data test-fields default)
          data      (merge data i)]
      (is (= data {:name "Harry" :password "baz"}))))

  (testing "bad incremental fill"
    (let [default   {:id 1, :name "Harry", :password "bar"}
          i         {:password "baz", :id 3}
          data      (merge default (f/get-fields-data test-fields i))]
      ; the name should be nil as it is a form field but not provided in submit
      ; the id is retained as 1 as it is not a form field
      (is (= data {:id 1 :name nil :password "baz"}))))))


(deftest test-render-fields
  (testing "render textarea"
    (let [field (f/form-field "message" {:field-type ::c/textarea})]
      (is (= (str "<label for=\"id_message\">Message</label>"
                  "<textarea class=\"u-full-width\" id=\"id_message\" name=\"message\"></textarea>")
             (c/render field nil nil))))))


(deftest test-validate-fields
  (testing "validate email field"
    (let [fields  [(f/form-field "email")]
          data    {:email "foo"}
          errors  (f/checks (f/check-non-blank fields data)
                            :email  (v/email? (:email data)))]
      (is (= errors {:email "Please provide a valid email"}))))

  (testing "validate-fields all clean"
    (let [data    {:name "Foo", :password "bar"}
          errors  (f/checks (f/check-non-blank test-fields data)
                            ::f/form (name-is-foo? data))]
      (is (= (count test-fields) 2))
      (is (= errors :no-errors))))

  (testing "validate-fields without password"
    (let [data      {:name "Foo"}
          errors    (f/checks (f/check-non-blank test-fields data)
                              ::f/form (name-is-foo? data))]
      (is (= errors {:password "This is a required field"}))))

  (testing "validate-fields wrong name"
    (let [data    {:name "Wrong", :password "bar"}
          errors  (f/checks (f/check-non-blank test-fields data)
                            ::f/form (name-is-foo? data))]
      (is (= errors {::f/form "Please check the name"})))))
