(ns utilities.files
  (:require [clojure.string :as s]
            [clojure.java.io :as io]))


(defn split-path
  "Splits given file path to a vector"
  [f]
  (s/split f #"/"))


(defn folder-name
  "Returns name of last folder in path"
  [path]
  (last (split-path path)))


(defn file-name
  "Returns name of the file or the last folder name"
  [path]
  (.getName (io/file path)))


(defn name-ext
  "Gives [name ext] for the given file path"
  [f]
  (let [file-name (last (split-path f))
        p (s/split file-name #"\.")
        ext (last p)
        name (s/join "." (butlast p))]
    [name ext]))


(defn matches?
  "Returns true if the file or the path matches given names"
  [f & names]
  (let [names (set names)]
    (or (contains? names f)
        (contains? names (file-name f)))))


(defn path
  "Joins parts to a file path"
  [& parts]
  (let [p (flatten (map split-path parts))
        s (first p)
        r (remove #(= % "") (rest p))]
    (s/join "/" (concat [s] r))))


(defn exists
  "Returns true if the given file or directory exists"
  [path]
  (.exists (io/file path)))


(defn list-files
  "Returns names of files in the given directory or its sub-directories"
  [dir-path]
  (let [root  (io/file dir-path)
        root-path (.getPath root)]
    (seq (for [f (file-seq root)
               :when (.isFile f)]
           (s/replace (.getPath f) root-path "")))))


(defn delete-dir
  "Deletes given directory path"
  [dir]
  (let [root        (io/file dir)
        root-path   (.getPath root)
        contents    (file-seq root)
        sub-dirs    (remove #(or (.isFile %)
                                 (= (.getPath %) root-path))
                            contents)
        root-files  (filter #(and (.isFile %)
                                  (= (.getParent %) root-path))
                            contents)]
    (println "ROOT" root-path)
    (println "SUB" sub-dirs)
    (println "FILES" root-files)
    (doseq [d sub-dirs]
      (delete-dir (.getPath d)))
    (doseq [f root-files]
      (io/delete-file f))
    (io/delete-file dir)))


(defn copy-file
  "Copy file from src to destination"
  [src dest]
  (io/make-parents dest)
  (io/copy (io/file src) (io/file dest)))


(defn copy-dir
  "Copies given directory from source to destination"
  [src dest]
  (if (exists dest)
    (delete-dir dest))
  (doseq [f (list-files src)]
    (copy-file (path src f) (path dest f))))
