# Routing - defining website urls

We define our urls in `project/urls.clj` file.

Urls are vector of `:url-name "/path" view/function`

```clojure
(def urls [:home "/" users/index
           :login "/login/" users/login
           :logout "/logout/" users/logout])
```

## Routing utilities

We use `Compojure` for defining our website routes. However, the `compojure` syntax is a bit verbose. Also, `compojure` doesn't provide reverse-routing to generate a url from the url-name. We have created a small library in `routing` for handling reverse-routing.


### Defining URLs

We define the urls using a simple collection of `[:url-name "/path/" view/function]`. We can pass keyword arguments in urls the same way we pass placeholders in our queries and templates.

```clojure
(:require [books.views :as books])

(def urls [:home "/" users/index
           ; urls with keywords
           :read-book "/book/::name-slug/" books/open
           :edit-book "/book/::id/edit/" books/edit])
```

The view functions will be called with `request` map and the `keywords` provided in the urls.

```clojure
; in index function
(defn index [request] "...")

; in books/views.clj
(defn open [request name-slug] "...")

(defn edit [request id] "...")
```


### Reverse-routing

We would need to link to various urls in the application as the project grows. We should use reverse-routing for linking these urls instead of providing static links. The way all the urls are defined in one single place, which can be changed / updated without needing any other modification.

To enable reverse-routing, we need to call the `set-urls` method with the urls vector defined earlier.

```clojure
; in urls.clj

(def urls [...])

; enable reverse-routing
(set-urls urls)
```

The reverse urls are obtained using `get-url` function which requires only the `url name` and `keyword context`.

```clojure
(get-url :read-book {:name-slug "alice-in-wonderland"})
; /book/alice-in-wonderland/

(get-url :edit-book {:id 123})
; /book/13/edit/
```

### Related Reading

[Throwing Exceptions](exceptions.md): How to raise `:page-not-found` errors in views.