(ns utilities.templates
  (:require [clojure.string :as s]
            [digest]
            [hiccup.core :refer [html]]
            [hiccup.element :refer [link-to]]
            [ring.util.anti-forgery :refer [anti-forgery-field]]
            [utilities.settings :as settings]
            [utilities.files :as files]
            [clojure.java.io :as io]
            [utilities.exceptions :as ex]))


(defn create-hash-copy
  "Hashes given file in given directory and returns new hashed name"
  [f res-dir]
  (if-not settings/in-production?
    f
    (let [res-dir       (io/resource res-dir)
          [name ext]    (files/name-ext f)
          content       (slurp (io/file res-dir f))
          hash          (digest/md5 content)
          new-name      (str name "." (subs hash 0 10) "." ext)
          d-path        (io/file res-dir "hashes" new-name)]
      (io/make-parents d-path)
      (spit d-path content)
      (str "/static/hashes/" new-name))))


(defn get-embed-link
  "Hashes given css files and return html to include"
  [f]
  (let [[_ ext] (files/name-ext f)]
    (condp = ext
      "css" (str "<link href=\"" f "\" rel=\"stylesheet\">")
      "js"  (str "<script src=\"" f "\"></script>")
      "png" (str "<link rel=\"shortcut icon\" type=\"image/png\" href=\"" f "\">"))))


(defn hash-links
  "Hashes given files and returns embed html"
  ([coll-f] (hash-links coll-f "public"))
  ([coll-f res-dir]
   (for [f coll-f]
     (-> f
         (create-hash-copy res-dir)
         get-embed-link))))


(defn- render-flash
  "Generates HTML for flash map"
  [{:keys [klass msg]}]
  (str "<div class=\"alert " klass "\">"
       msg "</div>"))


(defn- get-placeholder-value
  "Returns placeholder value from given context.
  If placeholder is not in context, then '' is used.
  Inbuilt placeholders are in form of ::placeholder-
  Inbuilt placeholders provided are:
  ::csrf-token-
  ::flash-
  "
  [placeholder context]
  (cond
    (= placeholder :csrf-token-) (anti-forgery-field)
    (= placeholder :flash-) (if-let [flash (:flash (:request context))]
                              (render-flash flash)
                              "")
    :else (let [value (or (get context placeholder) "")]
            (if (coll? value)
              (s/join "\n" value)
              value))))


(defn break-text
  "Splits the text into a vector of strings and keywords.
  Keywords are placeholders"
  [text]
  (let [pattern   #"::([\w-]+)"
        ; splitting to -1 ensures empty last string even if pattern found in end
        parts     (s/split text pattern -1)
        keys      (for [[_ key] (re-seq pattern text)]
                    (keyword key))]
    (concat (interleave parts keys)
            [(last parts)])))


(defn- -break-file
  "Reads given file and breaks it into vector of strings and keywords.
  Optimized for production to read and break only once."
  [f]
  (if-let [f (io/resource f)]
    (break-text (slurp f))
    (ex/raise :template-not-found (str "Template " f " not found"))))

(def break-file (settings/memoize-in-prod -break-file))


(defn fill-broken-text
  [broken-text context]
  "Fills text broken into keywords and placeholders with context.
  Returns joined text with filled placeholder values."
  (s/join "" (for [part broken-text]
               (if (keyword? part)
                 (get-placeholder-value part context)
                 part))))


(defn fill-in-file
  "Fills the placeholders in given file template using context."
  [f context]
  (let [broken-text (break-file f)]
    (fill-broken-text broken-text context)))


(defn fill-in-text
  "Fills the placeholders in given text template using context.
  Placeholders are in form of ::placeholder.
  "
  [text context]
  (-> text
      break-text
      (fill-broken-text context)))
