(ns utilities.forms.validations
  (:require [utilities.forms.core :as f]
            [clojure.string :as s]))

; Sources: https://docs.djangoproject.com/en/2.1/_modules/django/core/validators/#validate_ipv46_address


(defn validate-email
  "Returns true if given value is an email else raises error.

  Check is based on bouncer.validators which checks if there's
  only a single '@' and at least one point after the '@'
  "
  [value]
  (if (and (= (f/validate-not-blank value) ::f/is-clean)
           (re-matches #"^[^@]+@[^@\\.]+[\\.].+" value))
    ::f/is-clean
    "Please provide a valid email"))


(defn validate-min-length
  "Returns true if the mininum length is satisfied"
  [length value]
  (if (>= (count value) length)
    ::f/is-clean
    (str "Value must be minimum " length " characters")))


(defn validate-strong-password
  "Returns true if the password is minimum 6 chars"
  [value]
  (if (= (validate-min-length 6 value) ::f/is-clean)
    ::f/is-clean
    "Password should be minimum 6 characters"))


(defn validate-integer-str
  "Returns true if the string is an integer"
  [value]
  (try
    (Integer. value)
    ::f/is-clean
    (catch NumberFormatException e
      "Please provide a integer")))


