(ns {{ name }}.server
  (:require [compojure.core :refer [GET routes]]
            [compojure.route :as route]
            [ring.util.response :refer [redirect]]
            [utilities.exceptions :refer [exceptions-middleware]]
            [ring.middleware.defaults :refer [wrap-defaults site-defaults]]
            [ring.middleware.json :refer [wrap-json-response]]
            [utilities.settings :as settings]
            [utilities.routing :as r]
            [{{ name }}.pages :refer [page-not-found]]
            [immutant.web :as web]
            [taoensso.timbre :as timbre]
            [utilities.logging :as logger]
    ; Setup urls by requiring it
            [{{ name }}.urls :refer [urls]]
            [utilities.db :as db]))


; compojure.routes returns ring-handler function
; path-handler function takes map of request, inspect the method and path
; and matches it with appropriate view function
(def handler-routes
  (let [users     (r/get-handlers)
        default   [; static routes
                   (route/resources "/static/" {:root "public"})
                   ; Redirect missing slashes
                   (GET ":page{/.+[^/]$}" [page] (redirect (str page "/") 301))
                   (route/not-found page-not-found)]
        handlers  (concat users default)]
    (apply routes handlers)))


; Middlewares are higher level functions that add functionality to handlers
; First argument to middleware is handler function.
; It should return new handler function that should call original handler
; eg: https://github.com/ring-clojure/ring/wiki/Concepts#middleware

; disabling csrf in testing
(def default-middlewares
  (if settings/in-testing?
    (assoc site-defaults :security false)
    site-defaults))


(def ring-handler
  (-> handler-routes
      (wrap-defaults default-middlewares)
      (wrap-json-response)
      (exceptions-middleware)))


(defn start-development
  "Starts development server which reloads modules on change.
  Also shows in-browser stacktrace.
  Info and above to console.
  Connection pooling for database."
  []
  (db/enable-pooling)
  (web/run-dmc ring-handler))


(defn start-production
  "Starts production server
  Sends info and above to console,
  Errors and above to email,
  Uses connection pooling for database."
  []
  ; Merging as console is activated by default
  (timbre/merge-config! {:appenders {:mail-errs (logger/mailer)
                                     :spit      (taoensso.timbre.appenders.core/spit-appender)}})
  (db/enable-pooling)
  (web/run ring-handler :host "0.0.0.0" :port 8080))


(defn stop-production
  "Stops server"
  []
  (web/stop))
