(ns utilities.auth
  (:require [clojure.string :as s]
            [buddy.hashers :as hashers]
            [ring.util.response :as resp]
            [buddy.core.mac :as mac]
            [buddy.core.codecs :as codecs]
            [clj-time.core :as t]
            [utilities.settings :as settings]))


(defn encrypt-password
  "Returns encrypted password
  Encryption is Django style"
  [pass]
  (let [password          (hashers/derive pass {:alg :pbkdf2+sha256})
        [_ salt n hash]   (s/split password #"\$")]
    (s/join "$" ["pbkdf2_sha256" n salt hash])))


(defn password-matches?
  "Returns true if the passwords match"
  [raw encrypted]
  (let [[_ n salt hash]   (s/split encrypted #"\$")
        encrypted         (s/join "$" ["pbkdf2+sha256" salt n hash])]
    (hashers/check raw encrypted)))


(defn login-user
  "Returns response map with the user-id associated to session"
  [key val request next-url]
  (let [session (assoc (:session request) key val)
        response (assoc (resp/redirect next-url 303) :session session)]
    response))


(defn logout-user
  "Returns response map with the user-id removed from session"
  [request next-url]
  (let [response (resp/redirect next-url 303)]
    (assoc response :session nil)))


(defn- days-since-2001
  "returns days number of days since 2001-1-1 for given time"
  [time]
  (t/in-days (t/interval (t/date-time 2001 1 1) time)))


(defn- get-hash
  "Generates 20 characters hash for value using secret"
  [val secret]
  (-> (mac/hash val {:key secret :alg :hmac+sha256})
      (codecs/bytes->hex)
      (subs 0 20)))


(defn generate-login-token
  "Generate login token which is timestamp-token
  Token is generated by hashing some unique user value which is bound
  to change after using the token. Eg last_login

  Timestamp is number of days since 2001 converted to base 36
  This gives us a 3 digit string until around 2121
  "
  [unique-val]
  (let [ts      (-> t/now
                    (days-since-2001)
                    (Integer/toString 36))
        secret  (str "login-token:" settings/salt)
        token   (get-hash unique-val secret)]
    (str ts "-" token)))


(defn valid-login-token?
  "Returns true if the token is is valid.
  Token is invalid after 1 day."
  [token unique-val]
  (let [[ts token]  (s/split (str token) #"-")
        ts          (BigInteger. ts 36)
        secret      (str "login-token:" settings/salt)
        n-token     (get-hash unique-val secret)
        ts-period   (- (days-since-2001 (t/now)) ts)]
    (and (< ts-period 1)
         (= token n-token))))
