# Generating HTML Content

We have a `utilities.templates` library for filling templates.

Our templates have only one syntax and that is of placeholders.

Template `example.html`

```html
<h1>Welcome</h1>

::content

<footer>Copyright 2019</footer>
```

We can fill the above template using `(fill-in-file "example.html" {:content "Hello World"})`.

```clojure
(require '[utilities.templates :as t])

(t/fill-in-file "example.html" {:content "hello world"})

; <h1>Welcome</h1>
;
; hello world
;
; <footer>Copyright 2019</footer>
```


## Using Components

We recommend a **component-driven-templating** approach for generating html. This means that repeatable pieces of our html content should be generated using Clojure. These should be separate from the other code. And these should be responsible for rendering a particular input and not computing the input.

The `render-form` component in `utilities.forms` is an example of a component. We recommend defining components in the `components.clj` file of each app.

### Combining with hiccup

We can use `hiccup` for generating html inside a component. Hiccup takes a vector of vectors and converts it to html:

```clojure
(html [:div.primary "Hello World"])

; converts to
; <div class="primary">Hello World</div>

(html [:ul (map (fn [i] [:li i])
                [1 2 3 4])])

; converts to
; <ul>
;     <li>1</li>
;     <li>2</li>
;     <li>3</li>
;     <li>4</li>
; </ul>
```

### Using templates and hiccup together

We can use `fill-in-file` together with components to create rich pages.

```clojure
(def books ["Foo" "Bar" "Baz" "Foo Bar"])

(defn link-books
  [book]
  (html [:a {:href "#"} book])

(fill-in-text "These are the books: ::books" {:books (map link-book books)})
; These are the books
; <a href="#">Foo</a>
; <a href="#">Bar</a>
; <a href="#">Baz</a>
; <a href="#">Foo Bar</a>
```

We use `link-books` component to render each individual book, and then filled it inside the larger text template.


## Organizing for nesting


Define different base functions in `src/project/pages.clj`:

```clojure
(ns paper-books.pages
  (:require [hiccup.core :refer [html]]
            [utilities.templates :as t]
            [clojure.java.io :as io]))


(def ^:private normal-page-css
  (t/hash-links ["css/normalize.css" "css/style.css"]))

(def ^:private dashboard-css
  (t/hash-links ["css/normalize.css" "css/dashboard.css"]))

(def ^:private dashboard-js
  (t/hash-links ["css/search.js" "css/dashboard.js"]))


(defn- get-content
  "Fills content template with given context"
  [template context]
  (let [template    (io/resource (io/file "html/content" template))]
    (t/fill-in-file template context)))


(defn normal
  base.html
  [request title template context]
  (t/fill-in-file "html/base.html"
                  {:title   title
                   :css     normal-page-css
                   :content (get-content template context)
                   :request request}))


(defn dashboard
  "Renders template using dashboard.html as base"
  [request title template context]
  (t/fill-in-file "html/base.html"
                  {:title       title
                   :css         dashboard-css
                   :js          dashboard-js
                   :navigation  (get-navigation request context)
                   :content     (get-content template context)
                   :request     request}))
```

Here we have created two base templates for normal rendering and for rendering content in a dashboard with navigation. The two pages use a common base html but with differnt contexts.

The base template for above can be something like: `resources/html/base.html`

```html
<!DOCTYPE html>
<html lang="en">

<head>
    <!-- Basic Page Needs -->
    <meta charset="utf-8">
    <title>::title</title>

    <!-- Mobile Specific Meta -->
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <!-- CSS -->
    ::css
</head>

<body id="top">

    <a href="/">Paper Books</a>

    <div class="container">
        ::navigation
        <div class="content">
            ::flash-
            ::content
            ::content-footer
        </div>
        ::page-footer
    </div>
    ::js
</body>

</html>
``` 


The `hash-links` function takes the list of css or js files, and generates hashed links in production. This provides a huge performance boost as we can set very long expiry headers. This optimisation was provided in Django using their `staticfiles` framework.


## Inbuilt Placeholders

The library automatically parses few inbuilt placeholders for `csrf-token-` and `flash-`.


## Performance

- `Hiccup` uses macros which [convert the vectors to html][hiccup-macro] during compilation instead of in run-time.
- The placeholder conversions for content and pages are `memoized` / cached in production. So these too are very efficient.

[hiccup-macro]: http://tonsky.me/blog/hiccup/




## Examples

A template for `login.html`

```html
<h1>User Login</h1>

<form method="post">
    ::csrf-token-
    ::form
    
    <button type="submit">Submit</button>
    
    Forgot password? <a href="/password/forgot/">Click here to reset</a>
</form>

Don't have an account? <a href="/register/">Register Here</a>
```

Corresponding code in our `app/view.clj`

```clojure
(fill-in-file
    "login.html"
    {:form-fields (html [:input {:name "username"}]
                        [:password {:name "password"
                                    :type "password"}])})

; returns the content of login.html as string with
; placeholders replaced with given html
```


----

## Why not using Stencil or mustache

**`Mustache` provides good options like quoting, unquoting and functional sections. Then why shouldn't we use Mustache?**

I think that "rich" html components provide lot of assistance such as tooltips, color-highlights, pluralisation, thousand-separtors in numbers etc. These require lots of "(if then else)". Using templating languages will either not provide these or will be very in-efficient with lot of new syntaxes. Placeholders are simple and enable *component-driven* functional development.