# Generating HTML Content

In Django we have a concept of "extending" templates. So there is base template where we create blocks - and then we have other html files which extends those base template and fills those blocks.

Within templates we have tags and filters. Tags are like `for`, `if`, `cycle`, `groupby`, `firstof`... and filters are like `capfirst`, `join`, `length`, `lower`...this is suddenly a new language inside templating. Also, all of this is very inefficient, so nested `if` and nested `for` are not exactly recommended.

In Clojure, the popular library is `hiccup`. It takes a vector of vectors and converts it to html:

```clojure
(html [:div.primary "Hello World"])

; converts to
; <div class="primary">Hello World</div>

(html [:ul (map (fn [i] [:li i])
                [1 2 3 4])])

; converts to
; <ul>
;     <li>1</li>
;     <li>2</li>
;     <li>3</li>
;     <li>4</li>
; </ul>
```

This is pretty easy and efficient because it provides all the high level Clojure functions to generate HTML. I use it mostly for generating dynamic components such as tables or navigation.

However, Hiccup seems overkill to me for static markup around the web-page. For example, if we are creating a login page, then the form fields are dynamic because they provide error messages and validation. However there is still other static HTML which might be too verbose to write using `hiccup`. For this purpose I create a small library which works as a placeholder filler. So we can create a `login.html` file as:

```html
<h1>User Login</h1>

<form method="post">
    ::csrf-token-
    ::form-fields
    
    <button type="submit">Submit</button>
    
    Forgot password? <a href="/password/forgot/">Click here to reset</a>
</form>

Don't have an account? <a href="/register/">Register Here</a>
```

The `template.clj` provides the function `fill-in-file` which will simply replace the `::placeholders` with the passed replacements.

```clojure
(fill-in-file
    "login.html"
    {:form-fields (html [:input {:name "username"}]
                        [:password {:name "password"
                                    :type "password"}])})

; returns the content of login.html as string with
; placeholders replaced with given html
```

The library automatically parses few inbuilt placeholders for `csrf-token-` and `flash-`.

----

## "Extending" or nesting templates

The above functionality solves the `tags` and `filters` problem for content html and component html. However, large sites often have different HTML templates which use different CSS stylesheets and JS scripts. We usually create a `base.html` in Django with common css for a consistent look and feel. Then we "extend" other templates from `base.html` for different presentation purposes: `wide.html`, `no-sidebar.html`.

In Jurassic, we can nest such pages using function as needed. This makes everything very explicit and performant.

Define different base functions in `src/project/pages.clj`:

```clojure
(ns paper-books.pages
  (:require [hiccup.core :refer [html]]
            [utilities.templates :as t]
            [clojure.java.io :as io]))


(def ^:private normal-page-css
  (t/hash-links ["css/normalize.css" "css/style.css"]))

(def ^:private dashboard-css
  (t/hash-links ["css/normalize.css" "css/dashboard.css"]))

(def ^:private dashboard-js
  (t/hash-links ["css/search.js" "css/dashboard.js"]))


(defn- get-content
  "Fills content template with given context"
  [template context]
  (let [template    (io/resource (io/file "html/content" template))]
    (t/fill-in-file template context)))


(defn normal
  base.html
  [request title template context]
  (t/fill-in-file "html/base.html"
                  {:title   title
                   :css     normal-page-css
                   :content (get-content template context)
                   :request request}))


(defn dashboard
  "Renders template using dashboard.html as base"
  [request title template context]
  (t/fill-in-file "html/base.html"
                  {:title       title
                   :css         dashboard-css
                   :js          dashboard-js
                   :navigation  (get-navigation request context)
                   :content     (get-content template context)
                   :request     request}))
```

Here we have created two base templates for normal rendering and for rendering content in a dashboard with navigation. The two pages use a common base html but with differnt contexts.

The base template for above can be something like: `resources/html/base.html`

```html
<!DOCTYPE html>
<html lang="en">

<head>
    <!-- Basic Page Needs -->
    <meta charset="utf-8">
    <title>::title</title>

    <!-- Mobile Specific Meta -->
    <meta name="viewport" content="width=device-width, initial-scale=1">

    <!-- CSS -->
    ::css
</head>

<body id="top">

    <a href="/">Paper Books</a>

    <div class="container">
        ::navigation
        <div class="content">
            ::flash-
            ::content
            ::content-footer
        </div>
        ::page-footer
    </div>
    ::js
</body>

</html>
``` 


The `hash-links` function takes the list of css or js files, and generates hashed links in production. This provides a huge performance boost as we can set very long expiry headers. This optimisation was provided in Django using their `staticfiles` framework.


----

## Performance

- `Hiccup` uses macros which [convert the vectors to html][hiccup-macro] during compilation instead of in run-time.
- The placeholder conversions for content and pages are `memoized` / cached in production. So these are pretty efficient too.

[hiccup-macro]: http://tonsky.me/blog/hiccup/

----

## Why not using Stencil or mustache

**`Mustache` provides good options like quoting, unquoting and functional sections. Then why shouldn't we use Mustache?**

I think that "rich" html components provide lot of assistance such as tooltips, color-highlights, pluralisation, thousand-separtors in numbers etc. These require lots of "(if then else)". Using templating languages will either not provide these or will be very in-efficient with lot of new syntaxes. Placeholders are simple.
